package br.com.esec.icpm.libs.signature.response.handler.batch;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Deque;
import java.util.LinkedList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import br.com.esec.icpm.libs.Server;
import br.com.esec.icpm.libs.signature.exceptions.SignatureException;
import br.com.esec.icpm.libs.signature.helper.DownloadSignatureHelper;
import br.com.esec.icpm.libs.signature.helper.RequestStatusHelper;
import br.com.esec.icpm.libs.signature.response.handler.BaseSignatureHandler;
import br.com.esec.icpm.libs.signature.response.handler.SignatureHandler;
import br.com.esec.icpm.mss.ws.BatchSignatureTIDsRespType;
import br.com.esec.icpm.mss.ws.DocumentSignatureStatusInfoType;
import br.com.esec.icpm.mss.ws.SignatureStandardType;
import br.com.esec.icpm.mss.ws.SignatureStatusRespType;
import br.com.esec.icpm.mss.ws.notification.BatchSignatureNotificationType;
import br.com.esec.icpm.server.factory.Status;
import br.com.esec.icpm.server.ws.ICPMException;

public abstract class BaseBatchSignatureHandler extends BaseSignatureHandler implements BatchSignatureHandler {

	private static Logger log = LoggerFactory.getLogger(BaseBatchSignatureHandler.class);

	protected int statusCode;
	protected String statusMessage;

	protected SignatureStandardType standard;

	private Deque<DocumentStatus> documentsStatus = new LinkedList<DocumentStatus>();

	public BaseBatchSignatureHandler(Server server, SignatureStandardType standard, long transactionId) {
		super(server, transactionId);
		this.standard = standard;
	}

	protected void update(BatchSignatureTIDsRespType signatureStatusResp) throws ICPMException {
		statusCode = signatureStatusResp.getStatus().getStatusCode();
		statusMessage = signatureStatusResp.getStatus().getStatusMessage();
		documentsStatus = parseDocumentStatus(signatureStatusResp.getDocumentSignatureStatus());
	}

	protected void update(BatchSignatureNotificationType signatureStatusResp) throws ICPMException {
		statusCode = signatureStatusResp.getStatus().getStatusCode();
		statusMessage = signatureStatusResp.getStatus().getStatusMessage();
		documentsStatus = parseDocumentStatus(signatureStatusResp.getSignatures());
	}

	@Override
	public SignatureHandler save(OutputStream out) throws IOException, SignatureException {
		success();
		
		DocumentStatus docStatus = documentsStatus.removeFirst();

		log.info("Saving document signature of document '" + docStatus.getName() + "'.");

		// throw status exception
		validateStatus(transactionId, docStatus.getStatusCode(), docStatus.getStatusMesssage());

		if (docStatus.getStatusCode() == Status.SIGNATURE_VALID.getCode()) {
			switch (standard) {
			case CADES:
				saveCades(out, docStatus.getSignature());
				break;
			case ADOBEPDF:
				throw new IllegalStateException("Error. To save with AdobePdf we need the stream to original document again.");
			case XADES:
				throw new IllegalStateException("Error. To save with XAdES we need the stream to original document again.");
			default:
				throw new IllegalStateException("Standard '" + standard + "' is unrecognizable.");
			}
		} else {
			throw new IllegalStateException("The status " + docStatus.getStatusCode() + " is unrecognizable in this case.");
		}

		return this;
	}

	@Override
	public SignatureHandler saveAttached(OutputStream out) throws IOException, SignatureException {
		success();
		
		DocumentStatus docStatus = documentsStatus.removeFirst();

		log.info("Saving attached signature of document '" + docStatus.getName() + "'.");

		// throw status exception
		validateStatus(transactionId, docStatus.getStatusCode(), docStatus.getStatusMesssage());

		if (docStatus.getStatusCode() == Status.SIGNATURE_VALID.getCode()) {
			DownloadSignatureHelper.downloadAttachedDocument(server, docStatus.docId, out);
		} else {
			throw new IllegalStateException("The status " + docStatus.getStatusCode() + " is unrecognizable in this case.");
		}

		return this;
	}
	
	@Override
	public SignatureHandler saveAttached(InputStream in, OutputStream out) throws IOException, SignatureException {
		success();
		
		DocumentStatus docStatus = documentsStatus.removeFirst();

		log.info("Saving attached signature of document '" + docStatus.getName() + "'.");

		// throw status exception
		validateStatus(transactionId, docStatus.getStatusCode(), docStatus.getStatusMesssage());

		if (docStatus.getStatusCode() == Status.SIGNATURE_VALID.getCode()) {
			switch (standard) {
			case CADES:
				saveCades(docStatus.getDocId(), in, out, docStatus.getSignature());
				break;
			case ADOBEPDF:
				saveAdobePdf(docStatus.getDocId(), in, out, docStatus.getSignature());
				break;
			case XADES:
				saveXades(docStatus.getDocId(), in, out, docStatus.getSignature());
				break;
			default:
				throw new IllegalStateException("Standard '" + standard + "' is unrecognizable.");
			}
		} else {
			throw new IllegalStateException("The status " + docStatus.getStatusCode() + " is unrecognizable in this case.");
		}

		return this;
	}

	@Override
	public SignatureHandler success() throws SignatureException {
		validateStatus(transactionId, statusCode, statusMessage);
		return this;
	}

	public SignatureInfos getDocInfos() {
		return new SignatureInfos(transactionId, getDocumentsStatus());
	}

	protected Deque<DocumentStatus> parseDocumentStatus(List<DocumentSignatureStatusInfoType> signatures) throws ICPMException {
		Deque<DocumentStatus> documentsStatus = new LinkedList<DocumentStatus>();
		for (DocumentSignatureStatusInfoType docStatusInfo : signatures) {
			final long docId = docStatusInfo.getTransactionId();
			final int statusCode = docStatusInfo.getStatus().getStatusCode();
			final String statusMesssage = docStatusInfo.getStatus().getStatusMessage();
			final String name = docStatusInfo.getDocumentName();

			InputStream signature = null;
			try {
				if (statusCode == Status.SIGNATURE_VALID.getCode()) {
					SignatureStatusRespType status = RequestStatusHelper.requestStatus(server, docId);
					signature = parseSignature(statusCode, status.getSignature());
				}
			} catch (IOException e) {
				throw new IllegalStateException(e);
			}

			documentsStatus.add(new DocumentStatus(docId, name, statusCode, statusMesssage, signature));
		}
		return documentsStatus;
	}

	protected Deque<DocumentStatus> getDocumentsStatus() {
		return documentsStatus;
	}

	public class DocumentStatus {

		private final long docId;
		private final String name;
		private final int statusCode;
		private final String statusMesssage;
		private final InputStream signature;

		public DocumentStatus(long docId, String name, int statusCode, String statusMesssage, InputStream signature) {
			this.docId = docId;
			this.name = name;
			this.statusCode = statusCode;
			this.statusMesssage = statusMesssage;
			this.signature = signature;
		}

		public String getName() {
			return name;
		}

		public int getStatusCode() {
			return statusCode;
		}

		public InputStream getSignature() {
			return signature;
		}

		public long getDocId() {
			return docId;
		}

		public String getStatusMesssage() {
			return statusMesssage;
		}

	}

	public class SignatureInfos {

		private final long transactionId;
		private final Deque<DocumentStatus> documentsStatus;

		public SignatureInfos(long transactionId, Deque<DocumentStatus> documentsStatus) {
			this.transactionId = transactionId;
			this.documentsStatus = documentsStatus;
		}

		public long getTransactionId() {
			return transactionId;
		}

		public Deque<DocumentStatus> getDocumentsStatus() {
			return documentsStatus;
		}

	}
}
