package br.com.esec.icpm.libs.signature.response.polling;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;

import br.com.esec.icpm.libs.Server;
import br.com.esec.icpm.libs.signature.helper.RequestBatchSignatureHelper;
import br.com.esec.icpm.libs.signature.helper.RequestStatusHelper;
import br.com.esec.icpm.libs.signature.response.Futures;
import br.com.esec.icpm.mss.ws.BatchSignatureTIDsRespType;
import br.com.esec.icpm.server.factory.Status;
import br.com.esec.icpm.server.ws.ICPMException;

/**
 * Service to check the status of a batch signature by polling.
 * 
 * @author Tales Porto (tporto@esec.com.br|talesap@gmail.com)
 */
public class BatchPollingService extends BasePollingService<BatchSignatureTIDsRespType> {
	
	private static Logger log = LoggerFactory.getLogger(RequestBatchSignatureHelper.class);
	
	private static BatchPollingService instance;
	
	public static BatchPollingService getInstance() {
		if (instance == null)
			instance = new BatchPollingService();
		return instance;
	}

	public ListenableFuture<BatchSignatureTIDsRespType> status(Server server, long transactionId) {
		return status(server, transactionId, false);
	}
	
	public ListenableFuture<BatchSignatureTIDsRespType> status(Server server, long transactionId, boolean justWaitForDocInfos) {
		if (Futures.containsKey(server, transactionId)) {
			return Futures.get(server,transactionId);
		}

		SettableFuture<BatchSignatureTIDsRespType> future = SettableFuture.create();
		Futures.put(server, transactionId, future);

		final Checker checker = new Checker(server, transactionId, justWaitForDocInfos);
		schedule(checker);

		return future;
	}

	private class Checker extends BaseChecker {

		private boolean justWaitForDocInfos;
		
		public Checker(Server server, long transactionId, boolean justWaitForDocInfos) {
			super(server, transactionId);
			this.justWaitForDocInfos = justWaitForDocInfos;
		}

		@Override
		protected void check(SettableFuture<BatchSignatureTIDsRespType> future) {
			try {
				BatchSignatureTIDsRespType signatureStatusResp = RequestStatusHelper.requestBatchStatus(server, transactionId);

				log.info("[TRANSACTION ID " + transactionId + "] Checking batch transaction ...");
				
				if (justWaitForDocInfos) {
					if (signatureStatusResp.getDocumentSignatureStatus() != null) {
						future.set(signatureStatusResp);
					} else {
						schedule(this, counter); // schedule to run again in 2 seconds
						log.info("[TRANSACTION ID " + transactionId + "] In progess yet. A next check was scheduled.");
					}
				} else {
					if (signatureStatusResp.getStatus().getStatusCode() == Status.TRANSACTION_IN_PROGRESS.getCode()) {
						schedule(this, counter); // schedule to run again in 2 seconds
						log.info("[TRANSACTION ID " + transactionId + "] In progess yet. A next check was scheduled.");
					} else {
						future.set(signatureStatusResp);
					}
				}
			} catch (ICPMException e) {
				future.setException(e);
			}
		}

	}
	
	private BatchPollingService() {
	}
}
