package br.com.esec.icpm.libs.signature.response.polling;

import java.util.concurrent.TimeUnit;

import com.google.common.util.concurrent.SettableFuture;

import br.com.esec.icpm.libs.Server;
import br.com.esec.icpm.libs.signature.response.CertillionThreadPool;
import br.com.esec.icpm.libs.signature.response.Futures;

/**
 * BASE Service to check the status of a signature by polling.
 * 
 * @author Tales Porto (tporto@esec.com.br|talesap@gmail.com)
 */
public abstract class BasePollingService<T> {

	protected final int WAIT_TIME = 2 * 1000; // 2s
	protected final int MULTIPLIER_FACTOR_WAIT_TIME = 2;

	protected void schedule(final BaseChecker checker) {
		schedule(checker, 0);
	}

	/**
	 * <p> This method use the WAIT_TIME and check the tries.
	 * <p> If tries is greater than 25, it applies the MULTIPLIER_FACTOR_WAIT_TIME.
	 * <p> If tries is greater than 50, it applies the 2 * MULTIPLIER_FACTOR_WAIT_TIME.
	 * <p> If tries is greater than 75, it applies the 3 * MULTIPLIER_FACTOR_WAIT_TIME.
	 * <p> ...
	 * 
	 * @param checker
	 * @param tries
	 */
	protected void schedule(final BaseChecker checker, int tries) {
		final int waitTime = WAIT_TIME + (WAIT_TIME * (MULTIPLIER_FACTOR_WAIT_TIME * (tries / 25)));
		CertillionThreadPool.getInstance().schedule(checker, waitTime, TimeUnit.MILLISECONDS);
	}

	protected abstract class BaseChecker implements Runnable {

		protected static final int MAX_EXECUTIONS = 1000; // It define the max execution of it

		protected Server server;
		protected long transactionId;

		protected int counter = 0;

		public BaseChecker(Server server, long transactionId) {
			this.server = server;
			this.transactionId = transactionId;
		}

		@SuppressWarnings("unchecked")
		@Override
		public void run() {
			SettableFuture<T> future = (SettableFuture<T>) Futures.get(server, transactionId);

			// No one waiting for it
			if (future == null || future.isDone() || future.isCancelled()) {
				Futures.remove(server, transactionId);
				return;
			}

			//
			if (counter == MAX_EXECUTIONS) {
				Futures.remove(server, transactionId);
				return;
			}

			check(future);

			counter++;
		}

		protected abstract void check(SettableFuture<T> future);
	}
}
