package br.com.esec.icpm.libs.signature.response.handler.simple;

import java.io.IOException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;

import br.com.esec.icpm.libs.Server;
import br.com.esec.icpm.libs.signature.response.CertillionThreadPool;
import br.com.esec.icpm.libs.signature.response.handler.SignatureAsynchHandler;
import br.com.esec.icpm.libs.signature.response.notify.NotifyService;
import br.com.esec.icpm.mss.ws.SignatureRespType;
import br.com.esec.icpm.mss.ws.notification.SignatureNotificationType;
import br.com.esec.icpm.server.ws.ICPMException;

/**
 * Response Simple Asynchronous Signature Handler With Notification.
 * 
 * @author Tales Porto (tporto@esec.com.br|talesap@gmail.com)
 */
public class SignatureSimpleAsynchWithNotifyHandler extends BaseSimpleSignatureHandler implements SignatureAsynchHandler {

	private static Logger log = LoggerFactory.getLogger(SignatureSimpleAsynchWithNotifyHandler.class);

	public SignatureSimpleAsynchWithNotifyHandler(Server server, long transactionId) {
		super(server, transactionId);
	}

	public SignatureSimpleAsynchWithNotifyHandler(Server server, SignatureRespType response) {
		super(server, response);
	}

	@Override
	public SignatureSimpleAsynchWithNotifyHandler waitTo() throws ICPMException, InterruptedException, TimeoutException {
		return waitTo(DEFAULT_TIMEOUT);
	}

	@Override
	public SignatureSimpleAsynchWithNotifyHandler waitTo(long timeout) throws ICPMException, InterruptedException, TimeoutException {
		Future<SignatureNotificationType> result = NotifyService.getInstance().simple(server, transactionId);
		try {
			SignatureNotificationType signatureNotification = result.get(timeout, TimeUnit.SECONDS);
			
			log.info("Simple signature arrived.");
			
			parseNotification(signatureNotification);
		} catch (TimeoutException e) {
			result.cancel(true);
			throw e;
		} catch (ExecutionException e) {
			throw new IllegalStateException(e);
		} catch (IOException e) {
			throw new IllegalStateException(e);
		}

		return this;
	}
	
	@Override
	public SignatureAsynchHandler addListener(final Listener listener) throws ICPMException, InterruptedException, TimeoutException {
		final ListenableFuture<SignatureNotificationType> result = NotifyService.getInstance().simple(server, transactionId);
		Futures.addCallback(result, new FutureCallback<SignatureNotificationType>() {
			@Override
			public void onSuccess(SignatureNotificationType signatureNotification) {
				try {
					log.info("Batch signature arrived.");
					SignatureSimpleAsynchWithNotifyHandler.this.parseNotification(signatureNotification);
					listener.onSuccess(SignatureSimpleAsynchWithNotifyHandler.this);
				} catch (IOException e) {
					throw new IllegalStateException(e);
				}
			}

			@Override
			public void onFailure(Throwable t) {
				listener.onFailed(t);
				
			}
		}, CertillionThreadPool.getInstance());
		
		return this;
	}

	private void parseNotification(SignatureNotificationType signatureNotification) throws IOException {
		if (signatureNotification.getStatus() != null) {
			statusCode = signatureNotification.getStatus().getStatusCode();
			statusMessage = signatureNotification.getStatus().getStatusMessage();
			signature = parseSignature(statusCode, signatureNotification.getSignature());
		}
	}
}
