package br.com.esec.icpm.libs.signature.response.handler.batch;

import java.io.IOException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;

import br.com.esec.icpm.libs.Server;
import br.com.esec.icpm.libs.signature.response.CertillionThreadPool;
import br.com.esec.icpm.libs.signature.response.handler.SignatureAsynchHandler;
import br.com.esec.icpm.libs.signature.response.notify.NotifyService;
import br.com.esec.icpm.mss.ws.BatchSignatureComplexDocumentRespType;
import br.com.esec.icpm.mss.ws.SignatureStandardType;
import br.com.esec.icpm.mss.ws.notification.BatchSignatureNotificationType;
import br.com.esec.icpm.server.factory.Status;
import br.com.esec.icpm.server.ws.ICPMException;

/**
 * Response Batch Asynchronous Signature Handler With Notification.
 * 
 * @author Tales Porto (tporto@esec.com.br|talesap@gmail.com)
 */
public class SignatureBatchAsynchWithNotifyHandler extends BaseBatchAsynchSignatureHandler implements SignatureAsynchHandler {

	private static Logger log = LoggerFactory.getLogger(SignatureBatchAsynchWithNotifyHandler.class);

	public SignatureBatchAsynchWithNotifyHandler(Server server, SignatureStandardType standard, BatchSignatureComplexDocumentRespType response) throws IOException {
		super(server, standard, response.getTransactionId());

		if (response.getStatus() != null) {
			int statusCode = response.getStatus().getStatusCode();
			String statusMessage = response.getStatus().getStatusMessage();
			if (statusCode != Status.REQUEST_OK.getCode()) {
				throw new IllegalStateException("Something bad happened. The signature response return eith status " + statusMessage + " (" + statusCode + ").");
			}
		}
	}

	public SignatureBatchAsynchWithNotifyHandler(Server server, SignatureStandardType standard, long transactionId) {
		super(server, standard, transactionId);
	}

	@Override
	public SignatureBatchAsynchWithNotifyHandler waitTo() throws ICPMException, InterruptedException, TimeoutException {
		return waitTo(DEFAULT_TIMEOUT);
	}

	@Override
	public SignatureBatchAsynchWithNotifyHandler waitTo(long timeout) throws ICPMException, InterruptedException, TimeoutException {
		Future<BatchSignatureNotificationType> result = NotifyService.getInstance().batch(server, transactionId);
		try {
			BatchSignatureNotificationType signatureNotification = result.get(timeout, TimeUnit.SECONDS);
			
			log.info("Batch signature arrived.");
			
			update(signatureNotification);
		} catch (TimeoutException e) {
			result.cancel(true);
			throw e;
		} catch (ExecutionException e) {
			throw new IllegalStateException(e);
		}

		return this;
	}
	
	@Override
	public SignatureAsynchHandler addListener(final Listener listener) throws ICPMException, InterruptedException, TimeoutException {
		final ListenableFuture<BatchSignatureNotificationType> result = NotifyService.getInstance().batch(server, transactionId);
		Futures.addCallback(result, new FutureCallback<BatchSignatureNotificationType>() {
			@Override
			public void onSuccess(BatchSignatureNotificationType signatureNotification) {
				try {
					log.info("Batch signature arrived.");
					update(signatureNotification);
					listener.onSuccess(SignatureBatchAsynchWithNotifyHandler.this);
				} catch (ICPMException e) {
					throw new IllegalStateException(e);
				}
			}

			@Override
			public void onFailure(Throwable t) {
				listener.onFailed(t);
				
			}
			
		}, CertillionThreadPool.getInstance());
		
		return this;
	}

}
