package br.com.esec.icpm.libs.signature.response.handler;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import javax.activation.DataHandler;

import org.apache.commons.io.IOUtils;

import br.com.esec.icpm.libs.Server;
import br.com.esec.icpm.libs.signature.helper.attach.CadesAttachSignatureHelper;
import br.com.esec.icpm.libs.signature.helper.attach.PdfAttachSignatureHelper;
import br.com.esec.icpm.libs.signature.helper.attach.XadesAttachSignatureHelper;
import br.com.esec.icpm.server.factory.Status;

/**
 * Base Response Signature Handler.
 * 
 * @author Tales Porto (tporto@esec.com.br|talesap@gmail.com)
 */
public abstract class BaseSignatureHandler implements SignatureHandler {

	protected long transactionId;
	protected Server server;

	public BaseSignatureHandler(Server server, long transactionId) {
		this.server = server;
		this.transactionId = transactionId;
	}

	/**
	 * Get the transaction id of signature.
	 * 
	 * @return transaction id
	 */
	public long transactionId() {
		return transactionId;
	}
	
	/**
	 * Save CAdES detached signature on the signedDocument stream. 
	 * 
	 * @param signedDocument
	 * @param signature
	 * @throws IOException
	 */
	protected void saveCades(OutputStream signedDocument, InputStream signature) throws IOException {
		IOUtils.copy(signature, signedDocument);
	}

	/**
	 * Save CAdES attached signature on the signedDocument stream.
	 *  
	 * @param originalDocument
	 * @param signedDocument
	 * @param signature
	 * @throws IOException
	 */
	protected void saveCades(InputStream originalDocument, OutputStream signedDocument, InputStream signature) throws IOException {
		saveCades(transactionId, originalDocument, signedDocument, signature);
	}
	
	/**
	 * Save XAdES attached signature on the signedDocument stream.
	 *  
	 * @param originalDocument
	 * @param signedDocument
	 * @param signature
	 * @throws IOException
	 */
	protected void saveXades(InputStream originalDocument, OutputStream signedDocument, InputStream signature) throws IOException {
		saveXades(transactionId, originalDocument, signedDocument, signature);
	}
	
	/**
	 * Save ADOBE PDF attached signature on the signedDocument stream.
	 *  
	 * @param originalDocument
	 * @param signedDocument
	 * @param signature
	 * @throws IOException
	 */
	protected void saveAdobePdf(InputStream originalPdf, OutputStream signedPdf, InputStream signature) {
		saveAdobePdf(transactionId, originalPdf, signedPdf, signature);
	}

	protected void saveCades(long docId, InputStream originalDocument, OutputStream signedDocument, InputStream signature) throws IOException {
		CadesAttachSignatureHelper.attach(docId, originalDocument, signature, signedDocument);
	}

	protected void saveXades(long docId, InputStream originalDocument, OutputStream signedDocument, InputStream signature) throws IOException {
		XadesAttachSignatureHelper.attach(docId, originalDocument, signature, signedDocument);
	}

	protected void saveAdobePdf(long docId, InputStream originalPdf, OutputStream signedPdf, InputStream signature) {
		PdfAttachSignatureHelper.attach(docId, originalPdf, signature, signedPdf);
	}

	protected InputStream parseSignature(int statusCode, final DataHandler signatureDataHandler) throws IOException {
		if (signatureDataHandler == null)
			return null;

		if (statusCode == Status.SIGNATURE_VALID.getCode()) {
			if (signatureDataHandler != null) {
				return new ByteArrayInputStream(IOUtils.toByteArray(signatureDataHandler.getInputStream()));
			}
		}
		return null;
	}

}