package br.com.esec.icpm.libs.signature;

import java.io.IOException;
import java.util.concurrent.TimeoutException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import br.com.esec.icpm.libs.signature.exceptions.SignatureException;
import br.com.esec.icpm.libs.signature.helper.RequestSimpleSignatureHelper;
import br.com.esec.icpm.libs.signature.response.handler.simple.BaseSimpleSignatureHandler;
import br.com.esec.icpm.libs.signature.response.handler.simple.SignatureSimpleAsynchHandler;
import br.com.esec.icpm.libs.signature.response.handler.simple.SignatureSimpleAsynchWithNotifyHandler;
import br.com.esec.icpm.libs.signature.response.handler.simple.SignatureSimpleSynchHandler;
import br.com.esec.icpm.mss.ws.SignatureRespType;
import br.com.esec.icpm.server.ws.ICPMException;

/**
 * Simple signature request handler.
 * 
 * @author Tales Porto (tporto@esec.com.br|talesap@gmail.com)
 */
public class SimpleSignatureRequest {

	private static Logger log = LoggerFactory.getLogger(SimpleSignatureRequest.class);

	private SignatureRequest request;

	private String message;
	
	public SimpleSignatureRequest(SignatureRequest request) {
		this.request = request;
	}
	
	/**
	 * Set the message to sign.
	 * @param message
	 * @return this
	 */
	public SimpleSignatureRequest message(String message) {
		this.message = message;
		return this;
	}
	
	/**
	 * Call a synchronous signature.
	 * 
	 * <p> Block the thread and the server waiting for a signature.
	 * 
	 * @return signature response handler
	 * @throws ICPMException
	 * @throws IOException
	 */
	public BaseSimpleSignatureHandler sign() throws SignatureException, IOException {
		log.info("Requesting asynch batch signature to '" + request.identifier + "'.");
		SignatureRespType response = RequestSimpleSignatureHelper.requestSynchSimpleSignature(request.server, request.identifier, message, request.policyType, request.testMode, request.options.certificateFilters, request.apId);
		
		return new SignatureSimpleSynchHandler(request.server, response);
	}

	/**
	 * Call asynchronous signature.
	 * 
	 * <p> Don't block the server. Use <a href="https://en.wikipedia.org/wiki/Polling_%28computer_science%29">polling mechanism</a> to check the response.
	 * 
	 * @return signature response handler
	 * @throws ICPMException
	 * @throws IOException
	 */
	public SignatureSimpleAsynchHandler asynchSign() throws SignatureException, IOException {
		log.info("Requesting asynch batch signature to '" + request.identifier + "'.");
		SignatureRespType response = RequestSimpleSignatureHelper.requestAsynchSimpleSignature(request.server, request.identifier, message, request.policyType, request.testMode, request.options.certificateFilters, request.apId);
		
		return new SignatureSimpleAsynchHandler(request.server, response);
	}

	/**
	 * Call asynchronous signature.
	 * 
	 * <p> Don't block the server. Use notify mechanism to receive the response.
	 * 
	 * @return signature response handler
	 * @throws ICPMException
	 * @throws IOException
	 */
	public SignatureSimpleAsynchWithNotifyHandler asynchSignWithNotify() throws SignatureException, IOException {
		log.info("Requesting asynch batch signature to '" + request.identifier + "'.");
		SignatureRespType response = RequestSimpleSignatureHelper.requestAsynchWithNotifySimpleSignature(request.server, request.identifier, message, request.policyType, request.testMode, request.options.certificateFilters, request.apId);

		return new SignatureSimpleAsynchWithNotifyHandler(request.server, response);
	}

	/**
	 * Block on a arbitrary transaction id waiting for this response.
	 * 
	 * <p> Don't block the server. It use <a href="https://en.wikipedia.org/wiki/Polling_%28computer_science%29">polling mechanism</a> to check the response.
	 * 
	 * @param transactionId
	 * @return signature response handler
	 * @throws ICPMException
	 * @throws InterruptedException
	 * @throws TimeoutException
	 */
	public SignatureSimpleAsynchHandler waitTo(long transactionId) throws ICPMException, InterruptedException, TimeoutException {
		return new SignatureSimpleAsynchHandler(request.server, transactionId).waitTo();
	}
}
