package br.com.esec.icpm.libs.signature;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import br.com.esec.icpm.libs.Certillion;
import br.com.esec.icpm.libs.Server;
import br.com.esec.icpm.server.ws.SignaturePolicyType;

/**
 * Common signature request handler.
 * 
 * @author Tales Porto (tporto@esec.com.br|talesap@gmail.com)
 */
public class SignatureRequest {

	private static Logger log = LoggerFactory.getLogger(SignatureRequest.class);

	String identifier;
	boolean testMode;
	SignaturePolicyType policyType = SignaturePolicyType.AD_RB;
	Server server;
	Long apId = 0l;

	SignatureOptions options = new SignatureOptions();

	public SignatureRequest(Server server) {
		this.server = server;
	}

	/**
	 * Set this signature as test mode. Then you can use certificates of us TEST
	 * certificate chain.
	 * 
	 * @return this
	 */
	public SignatureRequest testMode() {
		testMode = true;
		return this;
	}

	/**
	 * Set your AP ID.
	 * 
	 * @param apId
	 * @return this
	 */
	public SignatureRequest apId(Long apId) {
		this.apId = apId;
		return this;
	}

	/**
	 * Set the identifier of the user to sign this request.
	 * 
	 * @param identifier
	 * @return this
	 */
	public SignatureRequest toUser(String identifier) {
		this.identifier = identifier;
		return this;
	}

	/**
	 * Get a simple signature request handler. Simple signature is some text to
	 * sign. Not documents.
	 * 
	 * @return simple signature request handler
	 */
	public SimpleSignatureRequest simple() {
		return new SimpleSignatureRequest(this);
	}

	/**
	 * Get a batch signature request handler. Batch signature is some text to
	 * sign. Not documents.
	 * 
	 * @return batch signature request handler
	 */
	public BatchSignatureRequest batch() {
		return new BatchSignatureRequest(this);
	}

	/**
	 * Choose a AD-RB(ASSINATURA DIGITAL COM REFERÊNCIA BÁSICA) policy to the
	 * signature.
	 * 
	 * <p>
	 * The ITI define a AD-RB like <a href=
	 * "http://www.iti.gov.br/component/content/article/190-repositorio/artefatos-de-assinatura-digital/4728-assinatura-digital-com-referencia-basica">
	 * it</a>.
	 * 
	 * @return this
	 */
	public SignatureRequest adRb() {
		policyType = SignaturePolicyType.AD_RB;
		return this;
	}

	/**
	 * Choose a AD-RT(ASSINATURA DIGITAL COM REFERÊNCIA DE TEMPO) policy to the
	 * signature.
	 * 
	 * <p>
	 * The ITI define a AD-RT like <a href=
	 * "http://www.iti.gov.br/component/content/article/190-repositorio/artefatos-de-assinatura-digital/4726-assinatura-digital-com-referencia-de-tempo">
	 * it</a>.
	 * 
	 * @return this
	 */
	public SignatureRequest adRt() {
		policyType = SignaturePolicyType.AD_RT;
		return this;
	}

	/**
	 * Choose a AD-RC(ASSINATURA DIGITAL COM REFERÊNCIAS COMPLETAS) policy to
	 * the signature.
	 * 
	 * <p>
	 * The ITI define a AD-RC like <a href=
	 * "http://www.iti.gov.br/component/content/article/190-repositorio/artefatos-de-assinatura-digital/4730-assinatura-digital-com-referencias-completas">
	 * it</a>.
	 * 
	 * @return this
	 */
	public SignatureRequest adRc() {
		policyType = SignaturePolicyType.AD_RC;
		return this;
	}

	/**
	 * Choose a AD-RV(ASSINATURA DIGITAL COM REFERÊNCIAS PARA VALIDAÇÃO) policy
	 * to the signature.
	 * 
	 * <p>
	 * The ITI define a AD-RV like <a href=
	 * "http://www.iti.gov.br/component/content/article/190-repositorio/artefatos-de-assinatura-digital/4729-assinatura-digital-com-referencia-para-validacao">
	 * it</a>.
	 * 
	 * @return this
	 */
	public SignatureRequest adRv() {
		policyType = SignaturePolicyType.AD_RV;
		return this;
	}

	/**
	 * Choose a AD-RA(ASSINATURA DIGITAL COM REFERÊNCIAS PARA ARQUIVAMENTO)
	 * policy to the signature.
	 * 
	 * <p>
	 * The ITI define a AD-RA like <a href=
	 * "http://www.iti.gov.br/component/content/article/190-repositorio/artefatos-de-assinatura-digital/4731-assinatura-digital-com-referencia-para-arquivamento">
	 * it</a>.
	 * 
	 * @return this
	 */
	public SignatureRequest adRa() {
		policyType = SignaturePolicyType.AD_RA;
		return this;
	}

	/**
	 * Set to use the production server.
	 * <p>
	 * This choice requite a keystore. {@link Certillion#config()}
	 * <p>
	 * The production host is a DEFAULT value. {@link Server#PRODUCTION}
	 * 
	 * @return this
	 */
	public SignatureRequest useProductionServer() {
		this.server = Server.PRODUCTION;
		return this;
	}

	/**
	 * Set to use the server of test.
	 * <p>
	 * This server have many limitation. Check on us site. {@link Server#TEST}
	 * 
	 * @return this
	 */
	public SignatureRequest useTestServer() {
		this.server = Server.TEST;
		return this;
	}

	/**
	 * Set to use an arbitrary server.
	 * 
	 * @param host
	 *            eg.: certillion.companyx.com
	 * @param secure
	 *            use https or not
	 * @return this
	 */
	public SignatureRequest useArbitraryServer(String host, boolean secure) {
		this.server = new Server(host, secure);
		return this;
	}

	/**
	 * Same advanced options.
	 * <p>
	 * if you aren't a advanced user of digital signature don't set it.
	 * 
	 * @return this
	 */
	public SignatureOptions options() {
		return options;
	}

}
