package br.com.esec.icpm.libs.signature;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeoutException;

import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import br.com.esec.icpm.libs.signature.exceptions.SignatureException;
import br.com.esec.icpm.libs.signature.helper.MimeTypeConstants;
import br.com.esec.icpm.libs.signature.helper.RequestBatchSignatureHelper;
import br.com.esec.icpm.libs.signature.response.handler.batch.SignatureBatchAsynchHandler;
import br.com.esec.icpm.libs.signature.response.handler.batch.SignatureBatchAsynchWithNotifyHandler;
import br.com.esec.icpm.libs.signature.response.handler.batch.SignatureBatchSynchHandler;
import br.com.esec.icpm.mss.ws.BatchSignatureComplexDocumentRespType;
import br.com.esec.icpm.mss.ws.SignatureStandardType;
import br.com.esec.icpm.server.ws.ICPMException;

/**
 * Batch signature request handler.
 * 
 * @author Tales Porto (tporto@esec.com.br|talesap@gmail.com)
 */
public class BatchSignatureRequest {

	private static Logger log = LoggerFactory.getLogger(BatchSignatureRequest.class);

	private SignatureRequest request;

	private SignatureStandardType standard = SignatureStandardType.CADES;
	private String message;
	private List<Document> documents = new ArrayList<Document>();

	public BatchSignatureRequest(SignatureRequest request) {
		this.request = request;
	}

	/**
	 * Set the signature to use <a href="https://en.wikipedia.org/wiki/CAdES_%28computing%29">CAdES</a> or CMS Advanced Electronic Signatures.
	 * 
	 * <p> If you want to know <a href="https://tools.ietf.org/html/rfc5126">more</a>.
	 * 
	 * @return this
	 */
	public BatchSignatureRequest cades() {
		standard = SignatureStandardType.CADES;
		return this;
	}

	/**
	 * Set the signature to use <a href="https://en.wikipedia.org/wiki/XAdES">XAdES</a> or XML Advanced Electronic Signatures.
	 * 
	 * <p> If you want to know <a href="https://www.w3.org/TR/XAdES/">more</a>.
	 * 
	 * @return this
	 */
	public BatchSignatureRequest xades() {
		standard = SignatureStandardType.XADES;
		return this;
	}

	/**
	 * Set the signature to use ADOBE PDF.
	 * 
	 * <p> Note: This is not PAdES yet.
	 * 
	 * <p> If you want to know <a href="https://www.adobe.com/devnet-docs/acrobatetk/tools/DigSig/Acrobat_DigitalSignatures_in_PDF.pdf">more</a>.
	 * 
	 * @return this
	 */
	public BatchSignatureRequest adobePdf() {
		standard = SignatureStandardType.ADOBEPDF;
		return this;
	}

	/**
	 * Message to explain to user this batch of documents. Eg.: "Documentos do Processo de número 1532 - Pensão do Fulano"
	 * @param message
	 * @return this
	 */
	public BatchSignatureRequest message(String message) {
		this.message = message;
		return this;
	}

	/**
	 * Add a document on this batch.
	 * 
	 * @param name
	 * @param in
	 * @return this
	 */
	public BatchSignatureRequest document(String name, InputStream in) {
		this.documents.add(new Document(name, in));
		log.info("Document '" + name + "' added.");
		return this;
	}

	/**
	 * Add a document on this batch.
	 * 
	 * @param name
	 * @param in
	 * @param mymeType
	 * @return this
	 */
	public BatchSignatureRequest document(String name, InputStream in, String mymeType) {
		this.documents.add(new Document(name, in, mymeType));
		log.info("Document '" + name + "' added.");
		return this;
	}

	/**
	 * Call a synchronous signature.
	 * 
	 * <p> Block the thread and the server waiting for a signature.
	 * 
	 * @return signature response handler
	 * @throws ICPMException
	 * @throws IOException
	 */
	public SignatureBatchSynchHandler sign() throws SignatureException, IOException {
		log.info("Requesting asynch batch signature to '" + request.identifier + "' with standard '" + standard + "' and policy type '" + request.policyType + "'...");
		BatchSignatureComplexDocumentRespType response = RequestBatchSignatureHelper.requestSynchBatchSignature(request.server, request.identifier, message, standard,
				request.policyType, request.testMode, request.options.certificateFilters, documents, request.apId);

		return new SignatureBatchSynchHandler(request.server, standard, response);
	}

	/**
	 * Call asynchronous signature.
	 * 
	 * <p> Don't block the server. Use <a href="https://en.wikipedia.org/wiki/Polling_%28computer_science%29">polling mechanism</a> to check the response.
	 * 
	 * @return signature response handler
	 * @throws ICPMException
	 * @throws IOException
	 */
	public SignatureBatchAsynchHandler asynchSign() throws SignatureException, IOException {
		log.info("Requesting asynch batch signature to '" + request.identifier + "' with standard '" + standard + "' and policy type '" + request.policyType + "'...");
		BatchSignatureComplexDocumentRespType response = RequestBatchSignatureHelper.requestAsynchBatchSignature(request.server, request.identifier, message, standard,
				request.policyType, request.testMode, request.options.certificateFilters, documents, request.apId);

		return new SignatureBatchAsynchHandler(request.server, standard, response);
	}

	/**
	 * Call asynchronous signature.
	 * 
	 * <p> Don't block the server. Use notify mechanism to receive the response.
	 * 
	 * @return signature response handler
	 * @throws ICPMException
	 * @throws IOException
	 */
	public SignatureBatchAsynchWithNotifyHandler asynchSignWithNotify() throws SignatureException, IOException {
		log.info("Requesting asynch batch signature to '" + request.identifier + "' with standard '" + standard + "' and policy type '" + request.policyType + "'...");
		BatchSignatureComplexDocumentRespType response = RequestBatchSignatureHelper.requestAsynchWithNotifyBatchSignature(request.server, request.identifier, message, standard,
				request.policyType, request.testMode, request.options.certificateFilters, documents, request.apId);

		return new SignatureBatchAsynchWithNotifyHandler(request.server, standard, response);
	}

	/**
	 * Block on a arbitrary transaction id waiting for this response.
	 * 
	 * <p> Don't block the server. It use <a href="https://en.wikipedia.org/wiki/Polling_%28computer_science%29">polling mechanism</a> to check the response.
	 * 
	 * @param transactionId
	 * @return signature response handler
	 * @throws ICPMException
	 * @throws InterruptedException
	 * @throws TimeoutException
	 */
	public SignatureBatchAsynchHandler waitTo(long transactionId) throws ICPMException, InterruptedException, TimeoutException {
		return new SignatureBatchAsynchHandler(request.server, standard, transactionId).waitTo();
	}

	public class Document {

		private final String name;
		private final InputStream in;
		private final String mimeType;

		public Document(String name, InputStream in) {
			this.name = name;
			this.in = in;
			this.mimeType = MimeTypeConstants.getMimeType(FilenameUtils.getExtension(name));
			log.info("Document without mimetype. Considering mimetype '" + this.mimeType + "'.");
		}

		public Document(String name, InputStream in, String mimeType) {
			this.name = name;
			this.in = in;
			this.mimeType = mimeType;
		}

		public String getName() {
			return name;
		}

		public InputStream getIn() {
			return in;
		}

		public String getMimeType() {
			return mimeType;
		}

	}
}
