package br.com.esec.icpm.libs.signature;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import javax.ws.rs.client.Client;
import javax.ws.rs.client.Invocation.Builder;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.xml.ws.Service;

import org.apache.commons.io.IOUtils;
import org.reflections.Reflections;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import br.com.esec.icpm.libs.DefaultHttpClientBuilder;
import br.com.esec.icpm.libs.HttpClientBuilder;
import br.com.esec.icpm.libs.Server;
import br.com.esec.icpm.mss.rest.ApplicationProviderResource;
import br.com.esec.icpm.mss.ws.SignaturePortType;
import br.com.esec.icpm.signer.ws.soap.CxfUtils;

/**
 * @author Tales Porto (tporto@esec.com.br|talesap@gmail.com)
 */
public class ApWsFactory {

	private static Logger log = LoggerFactory.getLogger(ApWsFactory.class);

	private static ApWsFactory instance;

	public static ApWsFactory getInstance() {
		if (instance == null)
			instance = new ApWsFactory();
		return instance;
	}

	private HttpClientManager httpClientManager = new HttpClientManager();
	private WsdlManager wsdlManager = new WsdlManager();

	private ApWsFactory() {
	}

	private ThreadLocal<SignaturePortType> soapServices = new ThreadLocal<SignaturePortType>();

	public SignaturePortType getSoapService(Server server) {
		try {
			if (soapServices.get() == null) {
				soapServices.set(buildSoapService(server));
			}
			return soapServices.get();
		} catch (MalformedURLException e) {
			throw new IllegalStateException(e);
		} catch (URISyntaxException e) {
			throw new IllegalStateException(e);
		}
	}

	private ThreadLocal<ApplicationProviderResource> restServices = new ThreadLocal<ApplicationProviderResource>();

	public ApplicationProviderResource getRestService(Server server) {
		if (restServices.get() == null) {
			restServices.set(buildRestService(server));
		}
		return restServices.get();
	}

	private SignaturePortType buildSoapService(Server server) throws MalformedURLException, URISyntaxException {
		URL signatureRequestSoapWsUrl = wsdlManager.getWsdlUrl(server);
		Service signatureService = Service.create(signatureRequestSoapWsUrl, SignaturePortType.QNAME);
		final SignaturePortType port = signatureService.getPort(SignaturePortType.class);

		if (server.isSecure())
			CxfUtils.config(port);
		return port;
	}

	private ApplicationProviderResource buildRestService(Server server) {
		return new ApplicationProviderClient(httpClientManager.getClient(server).target(server.getRestWsUrl()));
	}
	
	/**
	 * Builder of http clients.
	 * 
	 * DEFAULT: DefaultHttpClientBuilder
	 * 
	 * @param httpClientBuilder
	 */
	public void useClientBuilder(HttpClientBuilder httpClientBuilder) {
		httpClientManager.useClientBuilder(httpClientBuilder);
	}

	private class WsdlManager {

		private Map<Server, URL> wsdls = new ConcurrentHashMap<Server, URL>();

		private URL getWsdlUrl(Server server) throws MalformedURLException, URISyntaxException {
			if (!wsdls.containsKey(server)) {
				// TODO: Acredito que deveria sincronizar no map para não baixar mais de uma vez o mesmo WSDL. O único beneficio seria uma leve melhora no desempenho assim.
				URL localSignatureRequestSoapWsdlUrl = downloadAndStoreWsdl(server);
				wsdls.put(server, localSignatureRequestSoapWsdlUrl);
			}
			return wsdls.get(server);
		}

		private URL downloadAndStoreWsdl(Server server) throws URISyntaxException {
			Client client = httpClientManager.getClient(server);

			Response response = null;
			try {
				final Builder request = client.target(server.getSoapWsUrl()).request();
				response = request.get();
				if (response.getStatusInfo().getFamily() != Status.Family.SUCCESSFUL)
					throw new IllegalStateException("Error " + response.getStatus() + " when try download wsdl on url '" + server.getSoapWsUrl() + "'.");
				InputStream wsdlDataIn = response.readEntity(InputStream.class);

				File tempFile = File.createTempFile("certillion", ".wsdl");
				final FileOutputStream out = new FileOutputStream(tempFile);
				try {
					IOUtils.copy(wsdlDataIn, out);
				} finally {
					IOUtils.closeQuietly(out);
					IOUtils.closeQuietly(wsdlDataIn);
				}
				return tempFile.toURI().toURL();
			} catch (IOException e) {
				throw new IllegalStateException();
			} finally {
				if (response != null)
					response.close();
			}
		}
	}

	private class HttpClientManager {

		{
			// find the correct implementation of the HttpClientBuilder
			Reflections reflections = new Reflections("br.com.esec.icpm");
			Set<Class<? extends HttpClientBuilder>> implementations = reflections.getSubTypesOf(HttpClientBuilder.class);

			implementations.remove(DefaultHttpClientBuilder.class);

			try {
				if (implementations.size() == 0) {
					httpClientBuilder = new DefaultHttpClientBuilder();
				} else if (implementations.size() == 1) {
					Class<? extends HttpClientBuilder> clazz = implementations.iterator().next();
					httpClientBuilder = clazz.newInstance();
					log.trace("Using http builder '" + clazz + "'.");
				} else {
					throw new IllegalStateException("We found more then one implementation of HttpClientBuilder. Please, cleck it. " + implementations);
				}

			} catch (InstantiationException e) {
				throw new IllegalStateException("Cannot instantiate the implementation of '" + HttpClientBuilder.class + "'.", e);
			} catch (IllegalAccessException e) {
				throw new IllegalStateException("Cannot instantiate the implementation of '" + HttpClientBuilder.class + "'.", e);
			}
		}

		private HttpClientBuilder httpClientBuilder;
		private Map<Server, Client> clients = new ConcurrentHashMap<Server, Client>();

		private Client getClient(Server server) {
			if (!clients.containsKey(server)) {
				clients.put(server, httpClientBuilder.getClient(server));
			}
			return clients.get(server);
		}

		/**
		 * Builder of http clients.
		 * 
		 * DEFAULT: DefaultHttpClientBuilder
		 * 
		 * @param httpClientBuilder
		 */
		public void useClientBuilder(HttpClientBuilder httpClientBuilder) {
			this.httpClientBuilder = httpClientBuilder;
			clients.clear(); // regenerate the clients
		}
	}

}
