package br.com.esec.icpm.libs;

import br.com.esec.icpm.libs.signature.response.CertillionThreadPool;
import br.com.esec.icpm.signer.security.SecurityConfig;

/**
 * Configurable of the Certillion Library.
 * 
 * @author Tales Porto (tporto@esec.com.br|talesap@gmail.com)
 */
public class RootConfigurable implements Configurable {

	private KeystoreConfigurable keystoreConfigurable = new KeystoreConfigurable();
	private TruststoreConfigurable truststoreConfigurable = new TruststoreConfigurable();
	private ServerConfigurable serverConfigurable = new ServerConfigurable();

	private Boolean allowAllHosts;
	private int threadPoolSize;

	public void done() {
		// Security
		keystoreConfigurable.done();
		truststoreConfigurable.done();
		SecurityConfig.setAllowAllHosts(allowAllHosts);
		SecurityConfig.validate();

		// Server
		serverConfigurable.done();

		// Thread pool
		CertillionThreadPool.setThreadPoolSize(threadPoolSize);
	}

	/**
	 * Set if the SSL allow all host or check the host with the certificate
	 * 
	 * DEFAULT: false
	 * 
	 * @param allowAllHosts
	 * @return this
	 */
	public RootConfigurable allowAllHosts(boolean allowAllHosts) {
		this.allowAllHosts = allowAllHosts;
		return this;
	}

	/**
	 * Set the thread pool size used by this library.
	 * 
	 * DEFAULT: 5
	 * 
	 * @param threadPoolSize
	 * @return this
	 */
	public RootConfigurable threadPoolSize(int threadPoolSize) {
		this.threadPoolSize = threadPoolSize;
		return this;
	}

	/**
	 * Config the keystore used to sign SOAP request and to SSL authentications.
	 * 
	 * @return keystore config
	 */
	public KeystoreConfigurable keystore() {
		return keystoreConfigurable;
	}

	/**
	 * Config the trustsotre used to SSL connections.
	 * 
	 * @return truststore config
	 */
	public TruststoreConfigurable truststore() {
		return truststoreConfigurable;
	}

	/**
	 * Config the server configurations. DEFAULT: PRODUCTION
	 * 
	 * @return server config
	 */
	public ServerConfigurable server() {
		return serverConfigurable;
	}

	public class KeystoreConfigurable implements Configurable {
		private String path;
		private String type;
		private String password;
		private String keyAlias;

		/**
		 * Path to the key store.
		 * 
		 * @param keystorePath
		 * @return
		 */
		public RootConfigurable path(String keystorePath) {
			this.path = keystorePath;
			return RootConfigurable.this;
		}

		/**
		 * Type of keystore (JKS ou PKCS12). DEFAULT: JKS
		 * 
		 * @param keystoreType
		 * @return
		 */
		public RootConfigurable type(String keystoreType) {
			this.type = keystoreType;
			return RootConfigurable.this;
		}

		/**
		 * Pass to open the keystore.
		 * 
		 * @param keystorePassword
		 * @return
		 */
		public RootConfigurable password(String keystorePassword) {
			this.password = keystorePassword;
			return RootConfigurable.this;
		}

		/**
		 * Alias of the key on the keystore. If the keystore has one key this
		 * value is optional.
		 * 
		 * @param keyAlias
		 * @return
		 */
		public RootConfigurable keyAlias(String keyAlias) {
			this.keyAlias = keyAlias;
			return RootConfigurable.this;
		}

		@Override
		public void done() {
			SecurityConfig.setKeystorePath(path);
			SecurityConfig.setKeystoreType(type);
			SecurityConfig.setKeystorePassword(password);
			SecurityConfig.setKeyAlias(keyAlias);
		}
	}

	public class TruststoreConfigurable implements Configurable {
		private String path;
		private String type;
		private String password;

		/**
		 * Path to the truststore.
		 * 
		 * @param truststorePath
		 * @return
		 */
		public RootConfigurable path(String truststorePath) {
			this.path = truststorePath;
			return RootConfigurable.this;
		}

		/**
		 * Type of truststore (DIR, JKS ou PKCS12). DEFAULT: DIR
		 * 
		 * @param truststoreType
		 * @return
		 */
		public RootConfigurable type(String truststoreType) {
			this.type = truststoreType;
			return RootConfigurable.this;
		}

		/**
		 * Pass to open the truststore
		 * 
		 * @param truststorePassword
		 * @return
		 */
		public RootConfigurable password(String truststorePassword) {
			this.password = truststorePassword;
			return RootConfigurable.this;
		}

		@Override
		public void done() {
			SecurityConfig.setTruststorePath(path);
			SecurityConfig.setTruststoreType(type);
			SecurityConfig.setTruststorePassword(password);
		}
	}

	public class ServerConfigurable implements Configurable {
		private Server server;

		/**
		 * Set to use the production server. This server need to put a keystore.
		 * The production host is a DEFAULT value. {@link Server#PRODUCTION}
		 * 
		 * @return
		 */
		public RootConfigurable useProduction() {
			this.server = Server.PRODUCTION;
			return RootConfigurable.this;
		}

		/**
		 * Set to use the server of test. This server have many limitation.
		 * Check on us site. {@link Server#TEST}
		 * 
		 * @return
		 */
		public RootConfigurable useTest() {
			this.server = Server.TEST;
			return RootConfigurable.this;
		}

		/**
		 * Set to use an arbitrary server.
		 * 
		 * @param host
		 * @param secure
		 * @return
		 */
		public RootConfigurable useArbitrary(String host, boolean secure) {
			this.server = new Server(host, secure);
			return RootConfigurable.this;
		}

		/**
		 * Set to use an arbitrary server.
		 * 
		 * @param host
		 * @param secure
		 * @param soapWsUri
		 * @param restWsUri
		 * @return
		 */
		public RootConfigurable useArbitrary(String host, boolean secure, String soapWsUri, String restWsUri) {
			this.server = new Server(host, secure, soapWsUri, restWsUri);
			return RootConfigurable.this;
		}

		@Override
		public void done() {
			if (server != null)
				Certillion.server = server;
		}
	}

	/**
	 * @deprecated Use {@link RootConfigurable#keystore()}
	 * @param keystorePath
	 * @return
	 */
	@Deprecated
	public RootConfigurable keystorePath(String keystorePath) {
		keystoreConfigurable.path(keystorePath);
		return this;
	}

	/**
	 * @deprecated Use {@link RootConfigurable#keystore()}
	 * @param keystoreType
	 * @return
	 */
	@Deprecated
	public RootConfigurable keystoreType(String keystoreType) {
		keystoreConfigurable.type(keystoreType);
		return this;
	}

	/**
	 * @deprecated Use {@link RootConfigurable#keystore()}
	 * @param keystorePassword
	 * @return
	 */
	@Deprecated
	public RootConfigurable keystorePassword(String keystorePassword) {
		keystoreConfigurable.password(keystorePassword);
		return this;
	}

	/**
	 * @deprecated Use {@link RootConfigurable#keystore()}
	 * @param keyAlias
	 * @return
	 */
	@Deprecated
	public RootConfigurable keyAlias(String keyAlias) {
		keystoreConfigurable.keyAlias(keyAlias);
		return this;
	}

	/**
	 * @deprecated Use {@link RootConfigurable#truststore()}
	 * @param truststorePath
	 * @return
	 */
	@Deprecated
	public RootConfigurable truststorePath(String truststorePath) {
		truststoreConfigurable.path(truststorePath);
		return this;
	}

	/**
	 * @deprecated Use {@link RootConfigurable#truststore()}
	 * @param truststoreType
	 * @return
	 */
	@Deprecated
	public RootConfigurable truststoreType(String truststoreType) {
		truststoreConfigurable.type(truststoreType);
		return this;
	}

	/**
	 * @deprecated Use {@link RootConfigurable#truststore()}
	 * @param truststorePassword
	 * @return
	 */
	@Deprecated
	public RootConfigurable truststorePassword(String truststorePassword) {
		truststoreConfigurable.password(truststorePassword);
		return this;
	}

	/**
	 * Set to use the production server. This server need to put a keystore. The
	 * production host is a DEFAULT value. {@link Server#PRODUCTION}
	 * 
	 * @deprecated Use {@link RootConfigurable#server()}
	 * @return
	 */
	@Deprecated
	public RootConfigurable useProductionServer() {
		serverConfigurable.useProduction();
		return this;
	}

	/**
	 * Set to use the server of test. This server have many limitation. Check on
	 * us site. {@link Server#TEST}
	 * 
	 * @deprecated Use {@link RootConfigurable#server()}
	 * @return
	 */
	@Deprecated
	public RootConfigurable useTestServer() {
		serverConfigurable.useTest();
		return this;
	}

	/**
	 * Set to use an arbitrary server.
	 * 
	 * @deprecated Use {@link RootConfigurable#server()}
	 * @param host
	 * @param secure
	 * @return
	 */
	@Deprecated
	public RootConfigurable useArbitraryServer(String host, boolean secure) {
		serverConfigurable.useArbitrary(host, secure);
		return this;
	}

	/**
	 * Set to use an arbitrary server.
	 * 
	 * @deprecated Use {@link RootConfigurable#server()}
	 * @param host
	 * @param secure
	 * @param soapWsUri
	 * @param restWsUri
	 * @return
	 */
	@Deprecated
	public RootConfigurable useArbitraryServer(String host, boolean secure, String soapWsUri, String restWsUri) {
		serverConfigurable.useArbitrary(host, secure, soapWsUri, restWsUri);
		return this;
	}
}
