package br.com.esec.icpm.libs.signature.response.handler;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;

import br.com.esec.icpm.libs.signature.exceptions.SignatureException;
import br.com.esec.icpm.libs.signature.response.ThreadPool;
import br.com.esec.icpm.libs.signature.response.polling.SimplePollingService;
import br.com.esec.icpm.mss.ws.SignatureRespType;
import br.com.esec.icpm.mss.ws.SignatureStatusRespType;
import br.com.esec.icpm.server.ws.ICPMException;

public class SignatureSimpleAsynchHandler extends BaseSignatureHandler implements SignatureAsynchHandler {

	private static Logger log = LoggerFactory.getLogger(SignatureSimpleAsynchHandler.class);
	
	private InputStream signature;
	private int statusCode;
	private String statusMessage;

	public SignatureSimpleAsynchHandler(String host, long transactionId) {
		super(host, transactionId);
	}

	public SignatureSimpleAsynchHandler(String host, SignatureRespType response) {
		super(host, response.getTransactionId());

		try {
			if (response.getStatus() != null) {
				statusCode = response.getStatus().getStatusCode();
				statusMessage = response.getStatus().getStatusMessage();
				signature = parseSignature(statusCode, response.getSignature());
			}
		} catch (IOException e) {
			throw new IllegalStateException(e);
		}
	}

	@Override
	public SignatureSimpleAsynchHandler waitTo() throws ICPMException, InterruptedException, TimeoutException {
		return waitTo(DEFAULT_TIMEOUT);
	}

	@Override
	public SignatureSimpleAsynchHandler waitTo(long timeout) throws ICPMException, InterruptedException, TimeoutException {
		Future<SignatureStatusRespType> result = SimplePollingService.getInstance().status(host, transactionId);
		try {
			SignatureStatusRespType signatureStatusResp = result.get(timeout, TimeUnit.SECONDS);
			
			log.info("Simple signature arrived.");
			
			parseStatus(signatureStatusResp);
		} catch (TimeoutException e) {
			result.cancel(true);
			throw e;
		} catch (ExecutionException e) {
			throw new IllegalStateException(e);
		} catch (IOException e) {
			throw new IllegalStateException(e);
		}

		return this;
	}

	@Override
	public SignatureAsynchHandler addListener(final Listener listener) throws ICPMException, InterruptedException, TimeoutException {
		final ListenableFuture<SignatureStatusRespType> result = SimplePollingService.getInstance().status(host, transactionId);
		Futures.addCallback(result, new FutureCallback<SignatureStatusRespType>() {
			@Override
			public void onSuccess(SignatureStatusRespType signatureStatus) {
				try {
					log.info("Batch signature arrived.");
					SignatureSimpleAsynchHandler.this.parseStatus(signatureStatus);
					listener.onSuccess(SignatureSimpleAsynchHandler.this);
				} catch (IOException e) {
					throw new IllegalStateException(e);
				}
			}

			@Override
			public void onFailure(Throwable t) {
				listener.onFailed(t);
				
			}
		}, ThreadPool.instance);
		
		return this;
	}
	
	@Override
	public SignatureHandler save(OutputStream out) throws IOException, SignatureException {
		validateStatus(transactionId, statusCode, statusMessage);
		
		saveCades(out, signature);
		
		return this;
	}

	@Override
	public SignatureHandler save(InputStream in, OutputStream out) throws IOException, SignatureException {
		validateStatus(transactionId, statusCode, statusMessage);
		
		saveCades(in, out, signature);
		
		return this;
	}
	
	private void parseStatus(SignatureStatusRespType signatureStatusResp) throws IOException {
		if (signatureStatusResp.getStatus() != null) {
			statusCode = signatureStatusResp.getStatus().getStatusCode();
			statusMessage = signatureStatusResp.getStatus().getStatusMessage();
			signature = parseSignature(statusCode, signatureStatusResp.getSignature());
		}
	}


}
