package br.com.esec.icpm.libs.signature.response.handler;

import java.io.IOException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;

import br.com.esec.icpm.libs.signature.response.ThreadPool;
import br.com.esec.icpm.libs.signature.response.polling.BatchPollingService;
import br.com.esec.icpm.mss.ws.BatchSignatureComplexDocumentRespType;
import br.com.esec.icpm.mss.ws.BatchSignatureTIDsRespType;
import br.com.esec.icpm.mss.ws.SignatureStandardType;
import br.com.esec.icpm.server.factory.Status;
import br.com.esec.icpm.server.ws.ICPMException;

public class SignatureBatchAsynchHandler extends BaseBatchAsynchSignatureHandler implements SignatureAsynchHandler {

	private static Logger log = LoggerFactory.getLogger(SignatureBatchAsynchHandler.class);

	public SignatureBatchAsynchHandler(String host, SignatureStandardType standard, BatchSignatureComplexDocumentRespType response) throws IOException {
		super(host, standard, response.getTransactionId());

		if (response.getStatus() != null) {
			int statusCode = response.getStatus().getStatusCode();
			String statusMessage = response.getStatus().getStatusMessage();
			if (statusCode != Status.REQUEST_OK.getCode()) {
				throw new IllegalStateException("Something bad happened. The signature response return eith status " + statusMessage + " (" + statusCode + ").");
			}
		}
	}

	public SignatureBatchAsynchHandler(String host, SignatureStandardType standard, long transactionId) {
		super(host, standard, transactionId);
	}

	@Override
	public SignatureBatchAsynchHandler waitTo() throws ICPMException, InterruptedException, TimeoutException {
		return waitTo(DEFAULT_TIMEOUT);
	}

	@Override
	public SignatureBatchAsynchHandler waitTo(long timeout) throws ICPMException, InterruptedException, TimeoutException {
		Future<BatchSignatureTIDsRespType> result = BatchPollingService.getInstance().status(host, transactionId);
		try {
			BatchSignatureTIDsRespType signatureStatusResp = result.get(timeout, TimeUnit.SECONDS);
			
			log.info("Batch signature arrived.");
			
			documentsStatus = parseDocumentStatus(signatureStatusResp.getDocumentSignatureStatus());
		} catch (TimeoutException e) {
			result.cancel(true);
			throw e;
		} catch (ExecutionException e) {
			throw new IllegalStateException(e);
		}

		return this;
	}

	@Override
	public SignatureAsynchHandler addListener(final Listener listener) throws ICPMException, InterruptedException, TimeoutException {
		final ListenableFuture<BatchSignatureTIDsRespType> result = BatchPollingService.getInstance().status(host, transactionId, true);
		Futures.addCallback(result, new FutureCallback<BatchSignatureTIDsRespType>() {
			@Override
			public void onSuccess(BatchSignatureTIDsRespType signatureStatus) {
				try {
					log.info("Batch signature arrived.");
					documentsStatus = SignatureBatchAsynchHandler.this.parseDocumentStatus(signatureStatus.getDocumentSignatureStatus());
					listener.onSuccess(SignatureBatchAsynchHandler.this);
				} catch (ICPMException e) {
					throw new IllegalStateException(e);
				}
			}

			@Override
			public void onFailure(Throwable t) {
				listener.onFailed(t);
			}
			
		}, ThreadPool.instance);
		
		return this;
	}
	
}
