package br.com.edsilfer.kotlin_support.extensions

import android.annotation.TargetApi
import android.os.Build
import android.support.v7.app.AppCompatActivity
import android.text.Html
import android.util.TypedValue
import android.view.WindowManager
import br.com.edsilfer.kotlin_support.R
import br.com.edsilfer.kotlin_support.commons.Constants
import br.com.edsilfer.kotlin_support.presenter.dialog.DialogInput
import com.afollestad.materialdialogs.MaterialDialog
import com.gc.materialdesign.views.ProgressBarIndeterminate

enum class DisplayContainer {DIALOG, SNACKBAR }

enum class LogLevel { WARNING, ERROR }

// UI/NOTIFICATIONS ================================================================================
fun AppCompatActivity.showInputDialog(title: String,
                                      hint: String,
                                      onAccept: (text: String) -> Unit = {},
                                      onCancel: (text: String) -> Unit = {},
                                      onDismiss: (text: String) -> Unit = {},
                                      inputText: String? = "",
                                      inputTextSize: Float = 16f) {
    DialogInput(this)
            .setDialogTitle(title)
            .setInputHint(hint)
            .onAccept(onAccept)
            .onCancel(onCancel)
            .onDismiss(onDismiss)
            .setInputText(inputText!!)
            .setInputTextSize(inputTextSize)
            .show()
}

fun AppCompatActivity.showUnderConstructionPopUp() {
    val value = TypedValue()
    theme.resolveAttribute(R.attr.colorPrimaryDark, value, true)
    MaterialDialog.Builder(this)
            .title(getString(R.string.str_dlg_under_construction_title))
            .titleColor(value.data)
            .content(getString(R.string.str_dlg_under_construction_content))
            .positiveText(getString(R.string.str_commons_button_okay))
            .positiveColor(value.data)
            .negativeText("")
            .show()
}

// STRING PARAMETERS
fun AppCompatActivity.showErrorPopUp(content: String) {
    runOnUiThread {
        MaterialDialog.Builder(this)
                .title(Html.fromHtml("<b><font color='#c53929'> ${getString(R.string.str_dlg_error_title)}</d></font>"))
                .content(content)
                .positiveText(getString(R.string.str_commons_button_okay))
                .positiveColor(resources.getColor(android.R.color.holo_red_dark))
                .show()
    }
}

fun AppCompatActivity.showWarningPopUp(content: String) {
    MaterialDialog.Builder(this)
            .title(Html.fromHtml("<b><font color='#FFCC00'>${getString(R.string.str_dlg_warning_title)}</d></font>"))
            .content(content)
            .positiveText(getString(R.string.str_commons_button_okay))
            .positiveColor(resources.getColor(R.color.rsc_dialog_warning))
            .show()
}

fun AppCompatActivity.showPopUp(
        title: String,
        content: String,
        negativeText: String = getString(R.string.str_commons_button_cancel),
        onPositive: () -> Unit = {}) {
    runOnUiThread {
        val value = TypedValue()
        theme.resolveAttribute(R.attr.colorPrimaryDark, value, true)
        MaterialDialog.Builder(this)
                .title(title)
                .titleColor(value.data)
                .content(content)
                .positiveText(getString(R.string.str_commons_button_okay))
                .positiveColor(value.data)
                .negativeColor(value.data)
                .negativeText(negativeText)
                .onPositive { materialDialog, dialogAction -> onPositive() }
                .show()
    }
}

@TargetApi(Build.VERSION_CODES.LOLLIPOP_MR1)
fun AppCompatActivity.paintStatusBar(color : Int) {
    window.clearFlags(WindowManager.LayoutParams.FLAG_TRANSLUCENT_STATUS)
    window.addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS)
    window.statusBarColor = resources.getColor(color)
}

// RESOURCE PARAMETERS =============================================================================
fun AppCompatActivity.showWarningPopUp(content: Int) {
    MaterialDialog.Builder(this)
            .title(Html.fromHtml("<b><font color='#FFCC00'>${getString(R.string.str_dlg_warning_title)}</d></font>"))
            .content(getString(content))
            .positiveText(getString(R.string.str_commons_button_okay))
            .positiveColor(resources.getColor(R.color.rsc_dialog_warning))
            .show()
}

fun AppCompatActivity.showErrorPopUp(content: Int) {
    MaterialDialog.Builder(this)
            .title(Html.fromHtml("<b><font color='#c53929'> ${getString(R.string.str_dlg_error_title)}</d></font>"))
            .content(getString(content))
            .positiveText(getString(R.string.str_commons_button_okay))
            .positiveColor(resources.getColor(android.R.color.holo_red_dark))
            .show()
}

fun AppCompatActivity.showPopUp(title: Int, content: Int) {
    val value = TypedValue()
    theme.resolveAttribute(R.attr.colorPrimaryDark, value, true)
    MaterialDialog.Builder(this)
            .title(getString(title))
            .titleColor(value.data)
            .content(getString(content))
            .positiveText(getString(R.string.str_commons_button_okay))
            .positiveColor(value.data)
            .negativeColor(value.data)
            .negativeText(getString(R.string.str_commons_button_cancel))
            .show()
}

// LOADING =========================================================================================
fun AppCompatActivity.showCircularProgressBar() {
    try {
        runOnUiThread {
            findViewById(R.id.circularProgressBar)!!
                    .animate()
                    .translationY(getResources().getDimension(R.dimen.rsc_circular_progress_bar_height) / 2)
                    .alpha(1f).duration = Constants.LoadingInterface.LAYOUT_PROGRESS_BAR_INDETERMINATE_CIRCULAR_CIRCULAR_DURATION.toLong()
        }
    } catch (e: Exception) {
        throw RuntimeException(
                Constants.EXCEPTION_LAYOUT_MISSING.replace(Constants.PLACEHOLDER, Constants.LoadingInterface.LAYOUT_PROGRESS_BAR_INDETERMINATE_CIRCULAR))
    }

}

fun AppCompatActivity.hideCircularProgressBar() {
    try {
        runOnUiThread {
            findViewById(R.id.circularProgressBar)!!
                    .animate().translationY(getResources().getDimension(R.dimen.rsc_circular_progress_bar_height) / 2)
                    .alpha(0f)
                    .duration = Constants.LoadingInterface.LAYOUT_PROGRESS_BAR_INDETERMINATE_CIRCULAR_CIRCULAR_DURATION.toLong()
        }
    } catch (e: Exception) {
        throw RuntimeException(
                Constants.EXCEPTION_LAYOUT_MISSING.replace(Constants.PLACEHOLDER, Constants.LoadingInterface.LAYOUT_PROGRESS_BAR_INDETERMINATE_CIRCULAR))
    }
}

fun AppCompatActivity.showIndeterminateProgressBar() {
    try {
        runOnUiThread {
            val progressBar = findViewById(R.id.progress_bar) as ProgressBarIndeterminate
            progressBar.visibility = ProgressBarIndeterminate.VISIBLE
        }
    } catch (e: Exception) {
        throw RuntimeException(
                Constants.EXCEPTION_LAYOUT_MISSING.replace(Constants.PLACEHOLDER, Constants.LoadingInterface.LAYOUT_PROGRESS_BAR_INDETERMINATE_CIRCULAR))
    }
}

fun AppCompatActivity.hideIndeterminateProgressBar() {
    try {
        runOnUiThread {
            val progressBar = findViewById(R.id.progress_bar) as ProgressBarIndeterminate
            progressBar.visibility = ProgressBarIndeterminate.GONE
        }
    } catch (e: Exception) {
        throw RuntimeException(
                Constants.EXCEPTION_LAYOUT_MISSING.replace(Constants.PLACEHOLDER, Constants.LoadingInterface.LAYOUT_PROGRESS_BAR_INDETERMINATE_CIRCULAR))
    }
}

// OTHER ===========================================================================================

