package br.com.carenet.poc.hapvida.utils;

import android.content.Context;

import org.eclipse.paho.client.mqttv3.MqttCallbackExtended;
import org.eclipse.paho.client.mqttv3.MqttClient;
import org.eclipse.paho.client.mqttv3.MqttConnectOptions;
import org.eclipse.paho.client.mqttv3.MqttException;
import org.eclipse.paho.client.mqttv3.MqttMessage;
import org.joda.time.DateTime;

import static br.com.carenet.poc.hapvida.utils.LogUtils.LOGD;


public class MqttExample {
    static MqttCallbackExtended mCallback;

    public static void testSend(Context context) throws Exception {
        String algorithm          = "ES256"; // RS256 or ES256
        String messageType        = "event";
        int numMessages           = 10;
        int tokenExpMins          = 10;

        MqttConnectOptions connectOptions = MqttHelper.createMqttConnectOptions(context, algorithm);
        MqttClient client                 = MqttHelper.createMqttClient();

        // Both connect and publish operations may fail. If they do, allow retries but with an
        // exponential backoff time period.
        long initialConnectIntervalMillis     = 500L;
        long maxConnectIntervalMillis         = 6000L;
        long maxConnectRetryTimeElapsedMillis = 900000L;
        float intervalMultiplier              = 1.5f;
        DateTime iat                          = new DateTime();
        long retryIntervalMs                  = initialConnectIntervalMillis;
        long totalRetryTimeMs                 = 0;

        mCallback = MqttHelper.createAndAttachCallback(client);

        while (!client.isConnected() && totalRetryTimeMs < maxConnectRetryTimeElapsedMillis) {
            try {
                client.connect(connectOptions);
            } catch (MqttException e) {
                int reason = e.getReasonCode();

                // If the connection is lost or if the server cannot be connected, allow retries, but with
                // exponential backoff.
                System.out.println("An error occurred: " + e.getMessage());
                if (reason == MqttException.REASON_CODE_CONNECTION_LOST
                        || reason == MqttException.REASON_CODE_SERVER_CONNECT_ERROR) {
                    System.out.println("Retrying in " + retryIntervalMs / 1000.0 + " seconds.");
                    Thread.sleep(retryIntervalMs);
                    totalRetryTimeMs += retryIntervalMs;
                    retryIntervalMs *= intervalMultiplier;
                    if (retryIntervalMs > maxConnectIntervalMillis) {
                        retryIntervalMs = maxConnectIntervalMillis;
                    }
                } else {
                    throw e;
                }
            }
        }

        String mqttTopic = MqttHelper.getTelemetryTopic();

        for (int i = 1; i <= numMessages; ++i) {
            String payload = String.format("%s-payload-%d", MqttHelper.getDeviceId(), i);
            LOGD("MqttExample", String.format(
                    "Publishing %s message %d/%d: '%s'\n",
                    messageType, i, numMessages, payload));

            // Refresh the connection credentials before the JWT expires.
            // [START iot_mqtt_jwt_refresh]
            long secsSinceRefresh = ((new DateTime()).getMillis() - iat.getMillis()) / 1000;
            if (secsSinceRefresh > (tokenExpMins * 60)) {
                LOGD("MqttExample", String.format("\tRefreshing token after: %d seconds\n", secsSinceRefresh));
                iat = new DateTime();
                switch (algorithm) {
                    case "RS256":
                        connectOptions.setPassword(
                                JwtUtil.createJwtRsa2(context, MqttHelper.getProjectId()).toCharArray());
                        break;
                    case "ES256":
                        throw new IllegalArgumentException(
                                "Not supported at this time (VSMORI)");
                    default:
                        throw new IllegalArgumentException(
                                "Invalid algorithm " + algorithm
                                        + ". Should be one of 'RS256' or 'ES256'.");
                }
                client.disconnect();
                client.connect();
                mCallback = MqttHelper.createAndAttachCallback(client);
            }
            // [END iot_mqtt_jwt_refresh]

            // Publish "payload" to the MQTT topic. qos=1 means at least once delivery. Cloud IoT Core
            // also supports qos=0 for at most once delivery.
            MqttMessage message = new MqttMessage(payload.getBytes());
            message.setQos(1);
            client.publish(mqttTopic, message);

            if (messageType.equals("event")) {
                // Send telemetry events every second
                Thread.sleep(1000);
            } else {
                // Note: Update Device state less frequently than with telemetry events
                Thread.sleep(5000);
            }
        }

        // Disconnect the client if still connected, and finish the run.
        if (client.isConnected()) {
            client.disconnect();
        }

        LOGD("MqttExample","Finished loop successfully. Goodbye!");
        // [END iot_mqtt_publish]
    }
}