package br.com.carenet.poc.hapvida.services;

import android.app.Activity;
import android.app.Notification;
import android.app.PendingIntent;
import android.app.Service;
import android.content.Intent;
import android.os.Binder;
import android.os.CountDownTimer;
import android.os.Handler;
import android.os.IBinder;
import android.os.Vibrator;

import org.eclipse.paho.android.service.MqttAndroidClient;
import org.eclipse.paho.client.mqttv3.IMqttActionListener;
import org.eclipse.paho.client.mqttv3.IMqttDeliveryToken;
import org.eclipse.paho.client.mqttv3.IMqttToken;
import org.eclipse.paho.client.mqttv3.MqttCallbackExtended;
import org.eclipse.paho.client.mqttv3.MqttConnectOptions;
import org.eclipse.paho.client.mqttv3.MqttException;
import org.eclipse.paho.client.mqttv3.MqttMessage;

import java.util.ArrayList;
import java.util.Date;

import br.com.carenet.poc.hapvida.R;
import br.com.carenet.poc.hapvida.utils.LogUtils;
import br.com.carenet.poc.hapvida.utils.MqttHelper;

import static br.com.carenet.poc.hapvida.utils.LogUtils.LOGD;
import static br.com.carenet.poc.hapvida.utils.LogUtils.LOGE;


// This service should be used to upload pending data
// at once, before starting a new route
public class UploadDataService extends Service {
    private static final String TAG = LogUtils.makeLogTag(UploadDataService.class);

    public static final String MQTT_STARTED       = "com.mqtt.service.started";
    public static final String MQTT_CONNECTED     = "com.mqtt.connected";
    public static final String MQTT_DISCONNECTED  = "com.mqtt.disconnected";
    public static final String MQTT_CONNECT_FAIL  = "com.mqtt.connect.failed";
    public static final String MQTT_DATA           = "com.mqtt.indata";
    public static final String APP_DATA            = "some_data";

    public static final String NOTIFICATIONS = "ar4.notification.RECEIVED";

    private static UploadDataService _instance;
    private static MqttAndroidClient client;
    private static Boolean mConnected = false;
    public static String ClientID;
    private MqttConnectOptions options;
    private MqttConnectOptions mqttConnectOptions;
    private ArrayList <String> subs = new ArrayList <>();
    private Activity mainActivity;

    String algorithm          = "ES256"; // RS256 or ES256

    Notification serviceRunningNotification;

    // Keep a interval for sending data, kind of allows connection to be alive
    Handler handler       = new Handler();
    Runnable runnableCode = new Runnable() {
        @Override
        public void run() {
            if ( _instance == null ) {
                return;
            }
            if ( client.isConnected() ) {
                Date now = new Date();
                publish( MqttHelper.getTelemetryTopic(), "Teste from UploadDataService, oh yeah!!"+now.toString() );
                handler.postDelayed( runnableCode, 10000 );
            } else {
                log( "mq not connected." );
            }
        }
    };

    public static UploadDataService getInstance() {
        return _instance;
    }

    @Override
    public void onCreate() {
        super.onCreate();
        LOGD(TAG,"OnCreate");
        try {
            options = MqttHelper.createMqttConnectOptions(getApplicationContext(),algorithm);
            client  = createMqttAndroidClient();
        } catch (Exception ex) {
            //What to do here?
        }
    }

    @Override
    public int onStartCommand( Intent intent, int flags, int startId ) {
        log( "start, connected: " + isConnected() );
        _instance = this;
        broadcastUpdate( MQTT_STARTED );
        return START_STICKY;
    }

    @Override
    public IBinder onBind( Intent intent ) {
        log( "bind, connected: " + isConnected() );
        log( "Instance == " + ( this == _instance ) );
        if (!isConnected()) {
            this.connect();
        }
        return null;
    }

    @Override
    public void onTaskRemoved( Intent rootIntent ) {
        log( "task removed" );
        mainActivity = null;
        restartServiceInBG();
        super.onTaskRemoved( rootIntent );
    }


    /**
     * Some attempt to re-establish connection when disconnected
     */
    public void restartService() {
        log( "stopself" );
        broadcastUpdate( MQTT_DISCONNECTED );
        client.unregisterResources();
        client.close();
        try {
            client.disconnect();
        } catch ( MqttException e ) {
            e.printStackTrace();
        }
        _instance = null;
        stopSelf();
        stopForeground( true );
    }

    /**
     * Vibration
     */
    public void vibrate( Long time ) {
        Vibrator vib = ( Vibrator ) getSystemService( VIBRATOR_SERVICE );
        vib.vibrate( time );
    }

    /**
     * MainActivity instance
     */
    public void setMainActivity( Activity _activity ) {
        mainActivity = _activity;
    }

    /**
     * Attempt for restarting service and trying to maintain connection
     */
    public void restartServiceInBG() {
        // restart the service with a notification
        Intent notificationIntent = new Intent( getApplicationContext(), this.getClass() );
        notificationIntent.addFlags( Intent.FLAG_ACTIVITY_CLEAR_TOP | Intent.FLAG_ACTIVITY_SINGLE_TOP );
        PendingIntent pendingIntent = PendingIntent.getActivity( this, 0, notificationIntent, 0 );

        serviceRunningNotification = new Notification.Builder( this )
                .setContentTitle( "Service" )
                // ConextText part bugs, Notification maysay disconnected despite valid connection and vice versa
                .setContentText( isConnected() ? "Connected" : "Disconnected" )
                .setSmallIcon( R.drawable.ic_extension_black_24dp)
                .setContentIntent( pendingIntent )
                .setTicker( "ticker ?" )
                .addAction(R.drawable.ic_new_releases_black_24dp,"Open",
                        PendingIntent.getActivity( getBaseContext(), 0,
                                new Intent( "android.intent.category.LAUNCHER" ).setClassName( "com.package", "com.package.MainActivity" ),
                                PendingIntent.FLAG_UPDATE_CURRENT ) )
                .build();

        startForeground( 10, serviceRunningNotification );
        connect();
        log( "Starting in BG" );
    }

    /**
     * Creates android client
     *
     * @return Android client
     */
    private MqttAndroidClient createMqttAndroidClient() {
        log( "Create new client" );
        client = new MqttAndroidClient( this, "tcp://URL:PORT", ClientID );
        return client;
    }

    /**
     * Disconnect from broker
     */
    public void disconnect() {
        log( "disconnect..." );
        try {
            client.unregisterResources();
            //client.close(); // causes exceptions
            client.disconnect();
            client = null;
        } catch ( MqttException e ) {
            log( "1disconnect exception: " + e.toString() );
        } catch ( NullPointerException ne ) {
            log( "2disconnect exception:: " + ne.toString() );
        }
        broadcastUpdate( MQTT_DISCONNECTED );
    }


    /**
     * Connect to broker.
     */
    public void connect( final MqttAndroidClient client, MqttConnectOptions options ) {
        log( "Connect" );
        try {
            if ( client != null ) {
                IMqttToken token = client.connect( options );
                client.setTraceEnabled( true );
                token.setActionCallback( actionCallback );
                client.setCallback( mqttCallback );
            }
        } catch ( MqttException e ) {
            //handle e
            LOGE( TAG, "connect: " + e.getMessage(), e );
        } catch ( NullPointerException ne ) {
            LOGE( TAG, "connect: " + ne.getMessage(), ne );
        } catch ( Exception uknw ) {
            LOGE( TAG, "connect: " + uknw.getMessage(), uknw );
        }
    }

    /**
     * Connect to broker.
     * If client is not null, it will be reused instead of creating new one
     */
    public void connect() {
        log( "Connect: " + client );
        if ( client == null ) {
            client = createMqttAndroidClient();
        }
        this.connect( client, options );
    }

    /**
     * On connect
     */
    private IMqttActionListener actionCallback = new IMqttActionListener() {
        @Override
        public void onSuccess( IMqttToken asyncActionToken ) {
            log( "onconnect" );
            broadcastUpdate( MQTT_CONNECTED );
        }
        @Override
        public void onFailure( IMqttToken asyncActionToken, Throwable exception ) {
            log( "onconnect fail " + exception.toString() );
            broadcastUpdate( MQTT_CONNECT_FAIL );
        }
    };

    /**
     * On connectionlost and messages received
     */
    private MqttCallbackExtended mqttCallback = new MqttCallbackExtended() {

        @Override
        public void connectComplete(boolean reconnect, String serverURI) {
            LOGD( TAG, "connectComplete");
        }

        @Override
        public void connectionLost( Throwable cause ) {
            broadcastUpdate( MQTT_DISCONNECTED );
            client.unregisterResources();

            if ( null != cause ) {
                LOGD( TAG, "connection lost: " + cause.getMessage() + " " );
                LOGE( TAG, "connectionLost: ", cause );
            }

            if ( _instance.mainActivity == null ) {
                stopForeground( true );
                vibrate( 100L );
            }

            new CountDownTimer( 2000, 1000 ) {
                public void onTick( long millisUntilFinished ) {}
                public void onFinish() {
                    restartServiceInBG();
                }
            }.start();
        }

        @Override
        public void messageArrived( String topic, MqttMessage message ) throws Exception {
            LOGD( TAG, topic + " " + message.toString() + " dupe: " + message.isDuplicate() );
        }

        @Override
        public void deliveryComplete( IMqttDeliveryToken token ) {
            log( "delivery complete " + token.toString() );
        }
    };

    /**
     * Broadcast updates
     *
     * @param action What to broadcast
     * @param topic Which topic received data
     * @param data What's the data
     */
    private void broadcastUpdate( String action, String topic, String data ) {
        final Intent intent = new Intent( action );
        intent.putExtra( "topic", topic );
        intent.putExtra( "data", data );
        sendBroadcast( intent );
    }

    /**
     * Broadcast updates
     * Also sets mConnected, since broadcastUpdate( ) with Action only
     * Indicates mostly connectivity states
     *
     * @param action
     */
    private void broadcastUpdate( String action ) {
        log( "Broadcast: " + action );
        final Intent intent = new Intent( action );
        if ( action.equals( MQTT_CONNECTED ) || action.equals( MQTT_DATA ) ) {
            mConnected = true;
        } else if ( action.equals( MQTT_CONNECT_FAIL ) || action.equals( MQTT_DISCONNECTED ) ) {
            mConnected = false;
        }
        sendBroadcast( intent );
    }

    /**
     * Is connected
     *
     * @return boolean
     */
    public boolean isConnected() {

        return mConnected;
    }

    /**
     * Subscribe to a topic
     *
     * @param topic String Topic to subscribe to
     */
    public void subscribe( String topic ) {
        try {
            client.subscribe( topic, 0 );
        } catch ( Exception e ) {
            log( "Subscribe exception: " + e.toString() );
        }
    }

    /**
     * Publish method
     *
     * @param topic
     * @param msg
     * @return
     */
    public boolean publish( String topic, String msg ) {
        if ( client == null ) {
            return false;
        }
        return publish( topic, msg, false );
    }

    /**
     * Publish method
     *
     * @param topic
     * @param msg
     * @param retain
     * @return
     */
    public boolean publish( String topic, String msg, boolean retain ) {
        if ( client == null ) {
            return false;
        }
        if ( client.isConnected() ) {
            MqttMessage mMsg = new MqttMessage( new byte[ 0 ] );
            if ( msg == null ) {
                mMsg.setRetained( true );
            } else {
                mMsg.setRetained( retain );
                mMsg.setPayload( msg.getBytes() );
            }
            try {
                client.publish( topic, mMsg );
                return true;
            } catch ( MqttException e ) {
                log( "Publish exception " + e.toString() );
            }
        }
        return false;
    }

    /**
     * Binder
     */
    public class LocalBinder extends Binder {
        public UploadDataService getService() {
            log( "getService: " + ( UploadDataService.this == _instance ) );
            return UploadDataService.this;
        }
    }

    /**
     * LOGDebug
     *
     * @param string
     */
    private static void log( String string ) {
        LOGD( TAG, "log: " + string );
    }
}