package br.com.carenet.poc.hapvida.services;

import android.app.IntentService;
import android.content.Context;
import android.content.Intent;
import android.support.v4.content.LocalBroadcastManager;
import android.util.Log;

import org.eclipse.paho.client.mqttv3.IMqttDeliveryToken;
import org.eclipse.paho.client.mqttv3.MqttCallbackExtended;
import org.eclipse.paho.client.mqttv3.MqttClient;
import org.eclipse.paho.client.mqttv3.MqttConnectOptions;
import org.eclipse.paho.client.mqttv3.MqttException;
import org.eclipse.paho.client.mqttv3.MqttMessage;

import br.com.carenet.poc.hapvida.enums.DataServiceStates;
import br.com.carenet.poc.hapvida.utils.LogUtils;
import br.com.carenet.poc.hapvida.utils.MqttHelper;

import static br.com.carenet.poc.hapvida.enums.DataServiceStates.BROADCAST_PARAM_PACKAGE_ID;
import static br.com.carenet.poc.hapvida.enums.DataServiceStates.STATE_ACTION_COMPLETE;
import static br.com.carenet.poc.hapvida.enums.DataServiceStates.STATE_ACTION_ERROR;
import static br.com.carenet.poc.hapvida.utils.LogUtils.LOGD;
import static br.com.carenet.poc.hapvida.utils.LogUtils.LOGE;


// This service should be used along a route is being executed
public class StreamingDataService extends IntentService {
    // Used to write to the system log from this class.
    private static final String TAG = LogUtils.makeLogTag(StreamingDataService.class);
    private boolean finished = false;
    private boolean success  = false;
    private int packageId    = -1;
    //@TODO: Adicionar um watchdog
    private MqttClient client;

    // Defines and instantiates an object for handling status updates.
    //private BroadcastNotifier mBroadcaster = new BroadcastNotifier(this);

    /**
     * An IntentService must always have a constructor that calls the super constructor. The
     * string supplied to the super constructor is used to give a name to the IntentService's
     * background thread.
     */
    public StreamingDataService() {
        super("StreamingDataService");
    }

    /**
     * In an IntentService, onHandleIntent is run on a background thread.  As it
     * runs, it broadcasts its current status using the LocalBroadcastManager.
     * @param workIntent The Intent that starts the IntentService. This Intent contains the
     * URL of the web site from which the RSS parser gets data.
     */
    @Override
    protected void onHandleIntent(Intent workIntent) {
        // wait
        long transmissionCheckIntervalMillis    = 100L;
        long maxTransmissionCheckIntervalMillis = 3000L;
        long currentCheckIntervalMillis         = 0;

        try {
            String data = workIntent.getStringExtra("data");
            packageId = workIntent.getIntExtra(BROADCAST_PARAM_PACKAGE_ID,-3);
            if (packageId == -3)
                throw new Exception("No package id received");
            updateStatus(DataServiceStates.STATE_ACTION_CONNECTING);
            sendData(getApplicationContext(),data);

            while (!finished || currentCheckIntervalMillis>maxTransmissionCheckIntervalMillis) {
                Thread.sleep(transmissionCheckIntervalMillis);
                currentCheckIntervalMillis += transmissionCheckIntervalMillis;
            }

            if (!finished || !success)
                updateStatus(STATE_ACTION_ERROR);
            else
                updateStatus(STATE_ACTION_COMPLETE);
        } catch (Exception ex) {
            // Critical exception
            updateStatus(STATE_ACTION_ERROR);
            LOGE(TAG,ex.getMessage());
            Log.d("%#%#%#%#%#%#","Enviou STATE_ACTION_ERROR");
        }

        try {
            if (client != null) {
                client.setCallback(null);
                client.disconnect();
            }
        } catch (Exception ignored) {
        }
    }

    static MqttCallbackExtended mCallback;

    public void sendData(Context context, String payload) throws Exception {
        String algorithm                  = "ES256"; // RS256 or ES256
        MqttConnectOptions connectOptions = MqttHelper.createMqttConnectOptions(context, algorithm);
        client = MqttHelper.createMqttClient();
        // Both connect and publish operations may fail. If they do, allow retries but with an
        // exponential backoff time period.
        long retryIntervalMs                  = 500L;
        long maxConnectRetryTimeElapsedMillis = 1500L;
        long totalRetryTimeMs                 = 0;

        attachCallback(client);

        while (!client.isConnected() && totalRetryTimeMs < maxConnectRetryTimeElapsedMillis) {
            try {
                client.connect(connectOptions);
            } catch (MqttException e) {
                int reason = e.getReasonCode();
                // If the connection is lost or if the server cannot be connected, allow retries
                LOGD(TAG,"An error occurred: " + e.getMessage());
                if (reason == MqttException.REASON_CODE_CONNECTION_LOST
                        || reason == MqttException.REASON_CODE_SERVER_CONNECT_ERROR) {
                    LOGD(TAG,"Retrying in " + retryIntervalMs / 1000.0 + " seconds.");
                    Thread.sleep(retryIntervalMs);
                    totalRetryTimeMs += retryIntervalMs;
                } else {
                    throw e;
                }
            }
        }

        String mqttTopic    = MqttHelper.getTelemetryTopic();
        MqttMessage message = new MqttMessage(payload.getBytes());
        message.setQos(1);

        updateStatus(DataServiceStates.STATE_ACTION_SENDING);
        client.publish(mqttTopic, message);
    }

    private void attachCallback(final MqttClient client) throws MqttException {
        mCallback = new MqttCallbackExtended() {
            @Override
            public void connectComplete(boolean b, String s) {
                Log.w(TAG,"connectComplete");
            }

            @Override
            public void connectionLost(Throwable cause) {
                if (!finished) {
                    finished = true;
                    Log.w(TAG,"connectionLost");
                }
            }

            @Override
            public void messageArrived(String topic, MqttMessage message) throws Exception {
            }

            @Override
            public void deliveryComplete(IMqttDeliveryToken iMqttDeliveryToken) {
                success  = true;
                finished = true;
           }
        };

        client.setCallback(mCallback);
    }

    private void updateStatus(String state) {
        Intent localIntent = new Intent(DataServiceStates.BROADCAST_ACTION)
                .putExtra(DataServiceStates.BROADCAST_PARAM_STATUS, state)
                .putExtra(DataServiceStates.BROADCAST_PARAM_PACKAGE_ID, packageId);
        LocalBroadcastManager.getInstance(this).sendBroadcast(localIntent);
    }
}