package br.com.carenet.poc.hapvida;

import android.support.annotation.NonNull;
import android.util.Log;

import com.biovotion.android.gaphandler.VsmParametersReader;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.List;

import ch.hevs.biovotion.vsm.core.VsmDevice;
import ch.hevs.biovotion.vsm.parameters.Parameter;
import ch.hevs.biovotion.vsm.parameters.ParameterController;
import ch.hevs.biovotion.vsm.parameters.ParameterListener;

/**
 * Created by imartsekha on 11/20/17.
 */

public class DeviceParameterManager implements VsmParametersReader {
    private static final String TAG = DeviceParameterManager.class.getSimpleName();
    private VsmDevice mDevice;

    public static final int PARAMETER_TYPE_CLEAR_WHITE_LIST = 48;
    public static final int PARAMETER_TYPE_REQUEST_GAP = 17;
    public static final int PARAMETER_TYPE_GAP_REQUEST_STATUS = 33;
    public List<GapRequestListener> gapListeners = new ArrayList<>();


    public DeviceParameterManager(VsmDevice device) {
        this.mDevice = device;
    }

    public boolean readParameter(int parameterId, final ParameterReadListener listener) {
        this.mDevice.parameterController().addListener(new ParameterListener() {
            @Override
            public void onParameterWritten(@NonNull ParameterController parameterController, int i) {
                mDevice.parameterController().removeListeners();
            }

            @Override
            public void onParameterWriteError(@NonNull ParameterController parameterController, int i, int i1) {
                mDevice.parameterController().removeListeners();
            }

            @Override
            public void onParameterRead(@NonNull ParameterController parameterController, @NonNull Parameter parameter) {
                if(listener != null) {
                    listener.onParameterRead(parameter);
                }
                mDevice.parameterController().removeListener(this);
            }

            @Override
            public void onParameterReadError(@NonNull ParameterController parameterController, int i) {
                if(listener != null) {
                    listener.onParameterRead(null);
                }
                mDevice.parameterController().removeListeners();
            }
        });

        return this.mDevice.parameterController().readRequest(parameterId);
    }

    public boolean writeParameter(int parameterId, byte value, final ParameterWriteListener listener) {
        this.mDevice.parameterController().addListener(new ParameterListener() {
            @Override
            public void onParameterWritten(@NonNull ParameterController parameterController, int i) {
                if(listener != null) {
                    listener.onParameterWrite(true);
                }
                mDevice.parameterController().removeListeners();
            }

            @Override
            public void onParameterWriteError(@NonNull ParameterController parameterController, int i, int i1) {
                if(listener != null) {
                    listener.onParameterWrite(false);
                }
                mDevice.parameterController().removeListeners();
            }

            @Override
            public void onParameterRead(@NonNull ParameterController parameterController, @NonNull Parameter parameter) {
                mDevice.parameterController().removeListeners();
            }

            @Override
            public void onParameterReadError(@NonNull ParameterController parameterController, int i) {
                mDevice.parameterController().removeListeners();
            }
        });

        Parameter parameter = Parameter.fromBytes(parameterId, new byte[]{value});
        return this.mDevice.parameterController().writeRequest(parameter);
    }

    @Override
    public void requestGap(int streamType, long lastCounter, long rangeSize,
                           final VsmParametersReader.ParameterReadListener parameterReadListener) {

        Log.d(TAG, "requestGap: streamType = " + streamType + " lastCounter " + lastCounter + " range " + rangeSize);

        for (GapRequestListener gapListener : gapListeners) {
            gapListener.onGapRequestStarted(streamType, lastCounter, rangeSize);
        }

        ByteBuffer gapPayload = ByteBuffer.allocate(9).order(ByteOrder.LITTLE_ENDIAN);
        gapPayload.put((byte) streamType);

        //Convert counter to 4 bytes
        ByteBuffer longBuffer = ByteBuffer.allocate(Long.BYTES);
        longBuffer.putLong(lastCounter);
        byte[] lasCounterArray = longBuffer.array();
        for (int i = 7; i >= 4; i--) {
            gapPayload.put(lasCounterArray[i]);
        }

        gapPayload.putInt((int)rangeSize);

        Parameter gapRequestParameter = Parameter.fromBytes(PARAMETER_TYPE_REQUEST_GAP, gapPayload.array());
        this.mDevice.parameterController().addListener(new ParameterListener() {
            @Override
            public void onParameterWritten(@NonNull ParameterController parameterController, int parameterId) {
                mDevice.parameterController().removeListener(this);
                Log.d(TAG, "requestGap onParameterWritten: ");
                parameterReadListener.onSuccess(parameterId);
            }

            @Override
            public void onParameterWriteError(@NonNull ParameterController parameterController, int parameter, int errorCode) {
                mDevice.parameterController().removeListener(this);
                Log.d(TAG, "requestGap onParameterWriteError: ");
                parameterReadListener.onFailure(errorCode);
            }

            @Override
            public void onParameterRead(@NonNull ParameterController parameterController, @NonNull Parameter parameter) {

            }

            @Override
            public void onParameterReadError(@NonNull ParameterController parameterController, int i) {

            }
        });
        this.mDevice.parameterController().writeRequest(gapRequestParameter);
    }

    @Override
    public void requestGapStatus(final VsmParametersReader.ParameterReadListener parameterReadListener) {
        this.mDevice.parameterController().addListener(new ParameterListener() {
            @Override
            public void onParameterWritten(@NonNull ParameterController parameterController, int i) {
                // not needed
            }

            @Override
            public void onParameterWriteError(@NonNull ParameterController parameterController, int i, int i1) {
                // not needed
            }

            @Override
            public void onParameterRead(@NonNull ParameterController parameterController, @NonNull Parameter parameter) {
                mDevice.parameterController().removeListener(this);
                Log.d(TAG, "gapStatus onParameterRead: ");
                parameterReadListener.onSuccess(parameter.value()[0]);
                if (parameter.value()[0] == 1) {
                    for (GapRequestListener gapListener : gapListeners) {
                        gapListener.onGapRequestFinished();
                    }
                }
            }

            @Override
            public void onParameterReadError(@NonNull ParameterController parameterController, int i) {
                mDevice.parameterController().removeListener(this);
                Log.d(TAG, "gapStatus onParameterReadError: ");
                parameterReadListener.onSuccess(i);
            }
        });
        this.mDevice.parameterController().readRequest(PARAMETER_TYPE_GAP_REQUEST_STATUS);
    }

    @Override
    public void requestLastCounter(int i, CounterReadListener counterReadListener) {

    }

    @Override
    public void requestEventsCount(int i, CounterReadListener counterReadListener) {

    }

    public void registerGapListener(GapRequestListener listener) {
        gapListeners.add(listener);
    }

    public void clearGapListeners(){
        gapListeners.clear();
    }

    public interface ParameterReadListener {
        void onParameterRead(Parameter parameter);
    }

    public interface ParameterWriteListener {
        void onParameterWrite(boolean success);
    }

    public interface GapRequestListener {
        void onGapRequestStarted(int type, long startPoint, long size);
        void onGapRequestFinished();
    }
}
