package br.com.carenet.poc.hapvida;

import android.bluetooth.BluetoothDevice;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.IBinder;
import android.support.annotation.NonNull;
import android.util.Log;
import android.widget.TabHost;

import com.biovotion.android.decoder.stream.StreamDecoder;
import com.biovotion.android.decoder.stream.StreamNativeListener;
import com.biovotion.android.gaphandler.GapHandler;
import com.biovotion.android.gaphandler.storage.simple.SimpleGapStorage;
import com.biovotion.android.gaphandler.storage.simple.SimpleLastCounterStorage;

import java.lang.reflect.Method;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.List;

import ch.hevs.biovotion.vsm.core.VsmConnectionState;
import ch.hevs.biovotion.vsm.core.VsmDevice;
import ch.hevs.biovotion.vsm.core.VsmDeviceListener;
import ch.hevs.biovotion.vsm.parameters.Parameter;
import ch.hevs.ble.lib.core.BleService;

import static android.content.Context.BIND_AUTO_CREATE;

/**
 * Created by imartsekha on 11/15/17.
 */

public class DeviceConnectManager {
    private final static String TAG = DeviceConnectManager.class.getSimpleName();

    private static final String PREFFS_GAPS_TAG = "GapsTag";

    private BleService bleService;
    private VsmDevice device;
    private DeviceParameterManager deviceParameterManager;
    private GapHandler gapHandler;
    private SimpleGapStorage gapStorage;
    private SimpleLastCounterStorage counterStorage;
    private List<StreamNativeListener> streamNativeListeners = new ArrayList<>();
    private StreamDecoder streamDecoder;
    private VsmDeviceListener deviceListener;
    private Context context;
    private final int timeout;
    private boolean isConnecting;
    private BluetoothDevice unpairingDevice;
    private boolean isUnpairing;
    private boolean isGapHandlerInialized;

    // Code to manage Service lifecycle.
    private ServiceConnection mServiceConnection = new ServiceConnection() {
        @Override
        public void onServiceConnected(ComponentName name, IBinder service) {
            // Get BleService
            bleService = ((BleService.LocalBinder) service).getService();

            // Enable mode with logs
            if (BuildConfig.DEBUG) {
                bleService.setVerbose(true);

            }

            // The shared BLE service is now connected. Can be used by the watch.
            device.setBleService(bleService);

            Log.d(TAG, "onServiceConnected: initialize BLE service");
            //BleServie require to use ONLY ApplicationContext
            if (!bleService.initialize(context.getApplicationContext()))
                Log.e(TAG, "Unable to initialize Bluetooth");

            // Automatically connects to the device upon successful start-up initialization
            Log.d(TAG, String.format("Connecting to %s from activity %s", device.descriptor(), this.toString()));
            connectToDevice(device);
        }

        @Override
        public void onServiceDisconnected(ComponentName name) {
            bleService = null;
        }
    };

    private VsmDeviceListener mInnerDeviceListener = new VsmDeviceListener() {
        @Override
        public void onVsmDeviceReady(@NonNull VsmDevice vsmDevice) {
            isConnecting = false;
        }

        @Override
        public void onVsmDeviceConnecting(@NonNull VsmDevice vsmDevice) {
            isConnecting = true;
        }

        @Override
        public void onVsmDeviceConnected(@NonNull VsmDevice vsmDevice, boolean b) {
            isConnecting = false;
            gapStorage = new SimpleGapStorage(vsmDevice.descriptor().name(), context.getSharedPreferences(PREFFS_GAPS_TAG, Context.MODE_PRIVATE));
            counterStorage = new SimpleLastCounterStorage(vsmDevice.descriptor().name(), context.getSharedPreferences(PREFFS_GAPS_TAG, Context.MODE_PRIVATE));
            gapHandler = new GapHandler(gapStorage, counterStorage, deviceParameterManager);
            isGapHandlerInialized = true;
        }

        @Override
        public void onVsmDeviceConnectionError(@NonNull VsmDevice vsmDevice, VsmConnectionState vsmConnectionState) {
            isConnecting = false;
        }

        @Override
        public void onVsmDeviceDisconnected(@NonNull VsmDevice vsmDevice, int i) {
//            isConnecting = false;
//            isGapHandlerInialized = false;
//            gapHandler.stopGapsResolve();
//            if (isUnpairing && unpairingDevice != null) {
//                forgotDevice(unpairingDevice);
//            }
        }
    };

    public DeviceConnectManager(Context context, VsmDevice device, int timeout) {
        this.context = context;
        this.device = device;
        this.timeout = timeout;
        this.deviceParameterManager = new DeviceParameterManager(device);
    }

    public void connect() {
        if (bleService != null) {
            connectToDevice(device);
        } else {
            Intent gattServiceIntent = new Intent(context, BleService.class);
            context.bindService(gattServiceIntent, mServiceConnection, BIND_AUTO_CREATE);
        }
        device.addListener(mInnerDeviceListener);
    }

    public DeviceParameterManager parameterManager() {
        return deviceParameterManager;
    }

    public GapHandler gapHandler() {
        return gapHandler;
    }

    private void connectToDevice(VsmDevice device) {
        isConnecting = bleService.connect(device.descriptor().address(), timeout);
    }

    public void unpair() {
        isUnpairing = true;
        this.unpairingDevice = bleService.getConnectedDevice();
        this.deviceParameterManager.writeParameter(DeviceParameterManager.PARAMETER_TYPE_CLEAR_WHITE_LIST, (byte) 1, null);
    }

    public void forgotDevice(){
        unpair();
        // Unpair
        BluetoothDevice device = unpairingDevice;
        try {
            Method m = device.getClass().getMethod("removeBond", (Class[]) null);
            m.invoke(device, (Object[]) null);
        } catch (Exception e) {
            Log.e(TAG, e.getMessage());
        }
        //Close all services
        disconnect();
        isUnpairing = false;
        this.unpairingDevice = null;
    }
    private void forgotDevice(BluetoothDevice device) {
        // Unpair
        try {
            Method m = device.getClass().getMethod("removeBond", (Class[]) null);
            m.invoke(device, (Object[]) null);
        } catch (Exception e) {
            Log.e(TAG, e.getMessage());
        }
        //Close all services
        disconnect();
        isUnpairing = false;
        this.unpairingDevice = null;
    }

    public void disconnect() {
        this.device.disconnect();
        this.isConnecting = false;
    }

    public void close() {
        try{
            this.context.unbindService(mServiceConnection);
            this.device.removeListener(this.deviceListener);
            this.device.removeListener(this.mInnerDeviceListener);
            this.device.streamController().removeListeners();
            this.device.parameterController().removeListeners();
//        this.streamDecoder.clearListeners();
            this.streamDecoder = null;
            this.deviceListener = null;
            this.bleService.close();
            this.deviceParameterManager = null;
            this.bleService = null;
            this.context = null;
            streamNativeListeners.clear();
        }catch (Exception e){
            Log.e(TAG,e.toString());
        }


    }

    public void bind(VsmDeviceListener deviceListener) {
        this.deviceListener = deviceListener;
        device.addListener(this.deviceListener);
    }

    public void unBind(VsmDeviceListener deviceListener) {
        device.removeListener(deviceListener);
    }

    public void bindStreamListener(StreamNativeListener streamNativeListener) {
        streamNativeListeners.add(streamNativeListener);
        if (streamDecoder != null) {
            streamDecoder.addStreamListener(streamNativeListener);
        }
    }

    public void runDataStream() {
        if (this.device.isConnected()) {
            // Read StreamVersion from device to create StreamDecoder. To receive StreamVersion we need request Iterface version wiht parameterId = 0
            // Stream version is third and fourth byte of buffer in LITTLE_ENDIAN byte order
            this.deviceParameterManager.readParameter(0, new DeviceParameterManager.ParameterReadListener() {
                @Override
                public void onParameterRead(Parameter parameter) {
                    ByteBuffer bb = ByteBuffer.wrap(parameter.value());
                    bb.order(ByteOrder.LITTLE_ENDIAN);

                    int parameterVersion = bb.getShort();
                    int streamVersion = bb.getShort();
                    int commandVersion = bb.getShort();
                    int fotaVersion = bb.getShort();

                    Log.d(TAG, "onParameterRead: streamVersion " + streamVersion);

//                    streamDecoder = new StreamDecoder(streamVersion);
                    streamDecoder = new StreamDecoder();
                    if(!streamNativeListeners.isEmpty()) {

                        for (StreamNativeListener streamNativeListener : streamNativeListeners) {
                            streamDecoder.addStreamListener(streamNativeListener);
                        }
                    }

                    device.streamController().addListener(streamDecoder);
                }
            });
        }
    }

    public boolean isConnected() {
        return this.device.isConnected();
    }

    public boolean isUnpairing() {
        return isUnpairing;
    }

    public boolean isConnecting() {
        return this.device.isConnected() ? false : isConnecting;
    }
}
