package br.com.caelum.srs.dao;

import br.com.caelum.srs.SpacedRepetition;
import br.com.caelum.srs.model.*;
import org.hibernate.Session;
import org.joda.time.DateTime;

import javax.enterprise.context.RequestScoped;
import javax.inject.Inject;
import java.util.*;

import static br.com.caelum.srs.model.CardAvailability.UNLIMITED;
import static java.lang.Math.max;
import static java.lang.Math.min;
import static java.util.Optional.ofNullable;
import static org.joda.time.DateTime.now;

@RequestScoped
public class CardDAO {

	public static final long MAX_FOR_TODAY = 100;

	private final Session session;
	private final CardAnswerDAO answers;

	/**
	 * @deprecated cdi eyes only
	 */
	protected CardDAO() {
		this(null, null);
	}

	@Inject
	public CardDAO(Session session, CardAnswerDAO answers) {
		this.session = session;
		this.answers = answers;
	}

	public void save(List<Card> cards) {
		cards.forEach(session::save);
	}

	public Card save(Card card) {
		session.save(card);
		return card;
	}

	public Card load(Long id) {
		return (Card) session.load(Card.class, id);
	}

	public Long getRemainingForRepetitionToday(long userId){
		long answersToday = repetitionsAnsweredToday(userId);

		long practiceForToday = min(MAX_FOR_TODAY, getTotalCardsForRepetitionToday(userId));
		long remainingForToday = max(MAX_FOR_TODAY - answersToday, 0);

		return min(remainingForToday, practiceForToday);
	}

	public Long repetitionsAnsweredToday(long userId){
		return answers.repetitionsAnsweredToday(userId);
	}

	public Long getTotalCardsForRepetitionToday(long userId){
		return (Long) session.createQuery("select count(*) from Card c where c.studentId = :studentId "
				+ "and nextDisplayDate < :tomorrowMidnight").setParameter("studentId", userId)
				.setParameter("tomorrowMidnight", tomorrowMidnight()).uniqueResult();
	}


	public Long getNext(long userID, CardAvailability availability){
		if(UNLIMITED.equals(availability)){
			return getNextFor(userID);
		}

		if(answers.repetitionsAnsweredToday(userID) >= MAX_FOR_TODAY){
			return null;
		}

		return getNextFor(userID);
	}

	private Long getNextFor(long userID) {

		Optional<Card> answered = Optional.ofNullable((Card) session
			.createQuery("from Card c where c.studentId = :studentId "
			+ "and nextDisplayDate <= :tomorrowMidnight "
			+ "order by nextDisplayDate, id")
			.setParameter("studentId", userID)
			.setParameter("tomorrowMidnight", tomorrowMidnight())
			.setMaxResults(1).uniqueResult());

		return answered.map(c -> c.getRepetitionId()).orElse(null);
	}

	public DateTime lastAnswerFromTodayOrElse(Long studentId, DateTime fallback) {
		return ofNullable((DateTime) session
				.createQuery("select nextDisplayDate from Card "
					+ "where nextDisplayDate < :tomorrowMidnight and studentId=:studentId")
				.setParameter("tomorrowMidnight", tomorrowMidnight())
				.setParameter("studentId", studentId)
				.setMaxResults(1)
				.uniqueResult()).orElse(fallback);
	}

	private DateTime tomorrowMidnight() {
		return now().plusDays(1).toLocalDate().toDateTimeAtStartOfDay();
	}

	public Card getOrCreate(SpacedRepetition repetition, long userId) {
		return ofNullable((Card) session
				.createQuery("from Card where studentId = :studentId and repetitionId = :repetitionId")
				.setParameter("studentId", userId)
				.setParameter("repetitionId", repetition.getIdentifier())
				.setMaxResults(1)
				.uniqueResult())
				.orElseGet(() -> save(new Card(repetition, userId)));
	}
}
