package br.com.caelum.srs.model;

import java.util.List;

import static java.util.Arrays.asList;

public enum CardState {

	LEARNING_STEP1 {

		@Override
		public List<Difficulty> getDifficulties() {
			return asList(Difficulty.EASY, Difficulty.VERY_EASY);
		}

		@Override
		public Move move(Difficulty difficulty, Move previous) {
			if (difficulty.equals(Difficulty.BLACKOUT))
				return new Move(RECOVERING, 1);
			if (difficulty.equals(Difficulty.EASY))
				return new Move(LEARNING_STEP2, 10);
			if (difficulty.equals(Difficulty.VERY_EASY))
				return new Move(REVIEWING, FOUR_DAYS);
			throw new UnsupportedOperationException();
		}

		@Override
		public Difficulty difficultyFor(double difference) {
			if (difference < 0.15)
				return Difficulty.VERY_EASY;
			if (difference < 0.45)
				return Difficulty.EASY;
			return Difficulty.BLACKOUT;
		}
	},

	LEARNING_STEP2 {
		@Override
		public List<Difficulty> getDifficulties() {
			return asList(Difficulty.EASY, Difficulty.VERY_EASY);
		}

		@Override
		public Move move(Difficulty difficulty, Move previous) {
			if (difficulty.equals(Difficulty.BLACKOUT))
				return new Move(LEARNING_STEP1, 1);
			if (difficulty.equals(Difficulty.EASY))
				return new Move(REVIEWING, ONE_DAY);
			if (difficulty.equals(Difficulty.VERY_EASY)) {
				return new Move(REVIEWING, FOUR_DAYS);
			}
			throw new UnsupportedOperationException();
		}

		@Override
		public Difficulty difficultyFor(double difference) {
			if (difference < 0.15)
				return Difficulty.VERY_EASY;
			if (difference < 0.45)
				return Difficulty.EASY;
			return Difficulty.BLACKOUT;
		}
	},

	REVIEWING {
		@Override
		public List<Difficulty> getDifficulties() {
			return asList(Difficulty.HARD, Difficulty.EASY,
					Difficulty.VERY_EASY);
		}

		@Override
		public Move move(Difficulty difficulty, Move previous) {
			if (difficulty.equals(Difficulty.BLACKOUT))
				return new Move(RELEARNING, 10);

			if (difficulty.equals(Difficulty.HARD))
				return new Move(this, (int) (previous.getInterval() * 1.2),
						previous.getEasiness() * 0.85);

			if (difficulty.equals(Difficulty.EASY))
				return new Move(
						this,
						(int) (previous.getInterval() * previous.getEasiness()),
						previous.getEasiness());

			double nextEasiness = Math.min(previous.getEasiness() * 1.15,
					Move.MAX_EASINESS);
			return new Move(this,
					(int) (previous.getInterval() * nextEasiness),
					nextEasiness);

		}
	},

	RELEARNING {
		@Override
		public List<Difficulty> getDifficulties() {
			return asList(Difficulty.EASY);
		}

		@Override
		public Move move(Difficulty difficulty, Move previous) {
			if (difficulty.equals(Difficulty.BLACKOUT))
				return new Move(LEARNING_STEP1, 1);

			return new Move(REVIEWING, ONE_DAY);
		}

		@Override
		public Difficulty difficultyFor(double difference) {
			if (difference < 0.30)
				return Difficulty.EASY;
			return Difficulty.BLACKOUT;
		}
	},

	RECOVERING {

		@Override
		public List<Difficulty> getDifficulties() {
			return asList(Difficulty.EASY);
		}

		@Override
		public Move move(Difficulty difficulty, Move previous) {
			if (difficulty.equals(Difficulty.BLACKOUT))
				return new Move(this, 1);

			return new Move(LEARNING_STEP2, 10);
		}

		@Override
		public Difficulty difficultyFor(double difference) {
			if (difference < 0.30)
				return Difficulty.EASY;
			return Difficulty.BLACKOUT;
		}
	};

	private static int FOUR_DAYS = 4 * 24 * 60;
	private static int ONE_DAY = 24 * 60;

	public abstract List<Difficulty> getDifficulties();

	public abstract Move move(Difficulty difficulty, Move previous);

	public Difficulty difficultyFor(double difference) {
		if (difference < 0.15)
			return Difficulty.VERY_EASY;
		if (difference < 0.30)
			return Difficulty.EASY;
		if (difference < 0.45)
			return Difficulty.HARD;
		return Difficulty.BLACKOUT;
	}
}
