package br.com.caelum.srs.model;

import br.com.caelum.srs.SpacedRepetition;
import org.hibernate.annotations.Type;
import org.joda.time.DateTime;

import javax.persistence.*;
import java.util.Optional;

import static org.joda.time.DateTime.now;


@Entity
public class Card {

	static final int STREAK_TO_REMOVE = 12;

	@Id
	@GeneratedValue(strategy = GenerationType.IDENTITY)
	private Long id;

	private final Long studentId;

	private final Long repetitionId;

	@Type(type = "org.jadira.usertype.dateandtime.joda.PersistentDateTime")
	private final DateTime creationDate = now();

	@Type(type = "org.jadira.usertype.dateandtime.joda.PersistentDateTime")
	private DateTime nextDisplayDate = now();

	private int numberOfAnswers = 0;

	@Column(nullable = true)
	private int blackoutStreak = 0;

	@OneToOne
	private CardAnswer lastAnswer;
	public static final int INFINITE_YEARS = 2000;

	public Card(SpacedRepetition repetition, Long studentId) {
		this.repetitionId = repetition != null ? repetition.getIdentifier() : 0;
		this.studentId = studentId;
	}

	/**
	 * @deprecated hibernate only
	 */
	protected Card() {
		this(null, null);
	}

	public long nextIntervalInMinutes(Difficulty difficulty) {
		return getMove().move(difficulty).getInterval();
	}

	public CardAnswer answer(Difficulty difficulty, DateTime lastCardFromToday, AnswerKind answerKind) {
		Move currentMove = getMove();
		Move nextMove = currentMove.move(difficulty, blackoutStreak);

		this.numberOfAnswers = numberOfAnswers + 1;
		this.nextDisplayDate = lastCardFromToday.plusMinutes(nextMove.getInterval());
		if (nextDisplayDate.isBefore(now())) {
			this.nextDisplayDate = now().plusMinutes(nextMove.getInterval());
		}
		checkBlackoutStreak(difficulty);
		this.lastAnswer = new CardAnswer(this, difficulty, nextMove, answerKind);
		return lastAnswer;
	}

	private void checkBlackoutStreak(Difficulty difficulty) {
		if (Difficulty.BLACKOUT.equals(difficulty)) {
			blackoutStreak++;
		} else {
			blackoutStreak = 0;
		}
		if (blackoutStreak == STREAK_TO_REMOVE) {
			this.nextDisplayDate = now().plusYears(INFINITE_YEARS);
		}
	}

	public CardState getState() {
		return getMove().getState();
	}

	public LastAnswerResult getLastAnswerResult() {
		return Optional.ofNullable(this.lastAnswer)
			.map(c -> c.getDifficulty().getLastAnswerResult())
			.orElse(LastAnswerResult.NO_ANSWERS);
	}

	private Move getMove() {
		return Optional.ofNullable(lastAnswer)
				.map((answer) -> answer.getMove())
				.orElse(new Move(CardState.LEARNING_STEP1, 1));
	}

	public boolean isRemoved() {
		return nextDisplayDate.isAfter(now().plusYears(INFINITE_YEARS / 2));
	}

	public DateTime getNextDisplayDate() {
		return nextDisplayDate;
	}

	public Long getStudentId() {
		return studentId;
	}

	public Long getRepetitionId() {
		return repetitionId;
	}
}