package br.com.brjdevs.java.utils.extensions;

import br.com.brjdevs.java.utils.annotations.ExtensionClass;
import br.com.brjdevs.java.utils.annotations.MissingDocumentation;

import java.util.*;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@ExtensionClass
@MissingDocumentation
public class CollectionUtils {
	@SafeVarargs
	public static <T, U> Map<T, U> concat(Map<? extends T, ? extends U>... maps) {
		return Arrays.stream(maps)
			.flatMap(map -> map.entrySet().stream())
			.collect(Collectors.toMap(
				Map.Entry::getKey,
				Map.Entry::getValue
			));
	}

	@SafeVarargs
	public static <T, U> Map<T, U> concat(BinaryOperator<U> resolver, Map<? extends T, ? extends U>... maps) {
		return Arrays.stream(maps)
			.flatMap(map -> map.entrySet().stream())
			.collect(Collectors.toMap(
				Map.Entry::getKey,
				Map.Entry::getValue,
				resolver
			));
	}

	public static Iterable<String> iterable(Pattern pattern, String string) {
		return () -> {
			Matcher matcher = pattern.matcher(string);
			return new Iterator<String>() {
				@Override
				public boolean hasNext() {
					return matcher.find();
				}

				@Override
				public String next() {
					return matcher.group();
				}
			};
		};
	}

	public static <T, R> List<R> map(List<T> list, Function<T, R> mapper) {
		return list.stream().map(mapper).collect(Collectors.toList());
	}

	public static <T> T random(List<T> list, Random random) {
		return list.get(random.nextInt(list.size()));
	}

	public static <T> T random(T[] array, Random random) {
		return array[random.nextInt(array.length)];
	}

	public static <T> T random(List<T> list) {
		return list.get(random(list.size()));
	}

	public static <T> T random(T[] array) {
		return array[random(array.length)];
	}

	public static int random(int size) {
		return (int) Math.floor(Math.random() * size);
	}

	@SuppressWarnings("unchecked")
	public static <T> Comparator<T> randomOrder() {
		ThreadLocalRandom r = ThreadLocalRandom.current();
		int x = r.nextInt(), y = r.nextInt();
		boolean b = r.nextBoolean();
		return Comparator.comparingInt((T t) -> t.hashCode() ^ x)
			.thenComparingInt(s -> s.toString().length() ^ y)
			.thenComparing(((Comparator<T>) (b ? Comparator.naturalOrder() : Comparator.reverseOrder())));
	}

	@SafeVarargs
	public static <T> Stream<T> stream(T... objs) {
		return Stream.of(objs);
	}

	public static <T> List<T> subListOn(List<T> list, Predicate<T> predicate) {
		Optional<T> first = list.stream().filter(predicate).findFirst();
		return first.map(t -> list.subList(0, list.indexOf(t))).orElse(list);
	}

	public static <T> String toString(Collection<T> collection) {
		return toString(collection, Object::toString);
	}

	public static <T> String toString(Collection<T> collection, Function<T, String> toString) {
		return toString(collection, toString, "");
	}

	public static <T> String toString(Collection<T> collection, Function<T, String> toString, String join) {
		return collection.stream().map(toString).collect(Collectors.joining(join));
	}
}
