package br.com.brjdevs.java.utils.extensions;

import br.com.brjdevs.java.utils.annotations.ExtensionClass;
import br.com.brjdevs.java.utils.annotations.MissingDocumentation;
import br.com.brjdevs.java.utils.threads.CompletableThread;

import java.util.concurrent.*;
import java.util.function.Consumer;

@MissingDocumentation
@ExtensionClass
public class Async {
	public static Thread current() {
		return Thread.currentThread();
	}

	public static <T> Future<T> future(String task, Callable<T> callable) {
		return new CompletableThread<>(task, callable);
	}

	public static <T> Future<T> future(Callable<T> callable) {
		return new CompletableThread<>(callable);
	}

	public static void sleep(int milis) {
		try {
			Thread.sleep(milis);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * Start an Async single thread task every x seconds. Replacement for Timer.
	 *
	 * @param task         The name of the task to run
	 * @param scheduled    The runnable.
	 * @param everySeconds the amount of seconds the task will take to loop.
	 */
	public static void task(String task, Runnable scheduled, int everySeconds) {
		Executors.newSingleThreadScheduledExecutor(r -> new Thread(r, task)).scheduleAtFixedRate(scheduled, 0, everySeconds, TimeUnit.SECONDS);
	}

	/**
	 * Start an Async single thread task every x seconds. Replacement for Timer.
	 *
	 * @param task         The name of the task to run
	 * @param scheduled    The consumer that accepts the Executor as parameter.
	 * @param everySeconds the amount of seconds the task will take to loop.
	 */
	public static void task(String task, Consumer<ScheduledExecutorService> scheduled, int everySeconds) {
		ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor(r -> new Thread(r, task));
		executor.scheduleAtFixedRate(() -> scheduled.accept(executor), 0, everySeconds, TimeUnit.SECONDS);
	}

	/**
	 * Creates a thread with a Runnable and immediately starts the Thread
	 *
	 * @param doAsync the Runnable that the thread will run
	 * @return the Thread that is now executing the Runnable
	 */
	public static Thread thread(Runnable doAsync) {
		Thread thread = new Thread(doAsync);
		thread.start();
		return thread;
	}

	public static Thread thread(int sleepMilis, Runnable doAfter) {
		return thread(() -> {
			sleep(sleepMilis);
			if (doAfter != null) doAfter.run();
		});
	}

	public static Thread thread(String name, int sleepMilis, Runnable doAfter) {
		return thread(name, () -> {
			sleep(sleepMilis);
			if (doAfter != null) doAfter.run();
		});
	}

	/**
	 * Creates a thread with a Runnable and immediately starts the Thread
	 *
	 * @param name    The thread name
	 * @param doAsync The Runnable that the thread will run
	 * @return the Thread that is now executing the Runnable
	 */
	public static Thread thread(String name, Runnable doAsync) {
		Thread thread = new Thread(doAsync, name);
		thread.start();
		return thread;
	}
}