package br.com.brjdevs.java.utils.strings;

import java.io.BufferedReader;
import java.io.StringReader;
import java.util.*;
import java.util.regex.Pattern;

/**
 * Class made by AdrianTodt with a lot of useful and fast {@link String} and String[] utilities methods.
 */
public class StringUtils {
	public static final Pattern SPLIT_PATTERN = Pattern.compile("\\s+");

	//Short for:
	//advancedSplitArgsUnbox
	private static String advSplArgUnb(String s) {
		return s.replace("\\n", "\n").replace("\\r", "\r").replace("\\t", "\t").replace("\\\"", "\"").replace("\\\\", "\\");
	}

	public static String[] advancedSplitArgs(String args, int expectedArgs) {
		List<String> result = new ArrayList<>();
		boolean inAString = false;
		StringBuilder currentBlock = new StringBuilder();
		for (int i = 0; i < args.length(); i++) {
			if (args.charAt(i) == '"' && (i == 0 || args.charAt(i - 1) != '\\' || args.charAt(i - 2) == '\\'))
				inAString = !inAString;

			if (inAString)
				currentBlock.append(args.charAt(i));
			else if (Character.isSpaceChar(args.charAt(i))) {
				if (currentBlock.length() != 0) {
					if (currentBlock.charAt(0) == '"' && currentBlock.charAt(currentBlock.length() - 1) == '"') {
						currentBlock.deleteCharAt(0);
						currentBlock.deleteCharAt(currentBlock.length() - 1);
					}

					result.add(advSplArgUnb(currentBlock.toString()));
					currentBlock = new StringBuilder();
				}
			} else currentBlock.append(args.charAt(i));
		}

		if (currentBlock.length() != 0) {
			if (currentBlock.charAt(0) == '"' && currentBlock.charAt(currentBlock.length() - 1) == '"') {
				currentBlock.deleteCharAt(0);
				currentBlock.deleteCharAt(currentBlock.length() - 1);
			}

			result.add(advSplArgUnb(currentBlock.toString()));
		}

		String[] raw = result.toArray(new String[result.size()]);

		if (expectedArgs < 1) return raw;
		return normalizeArray(raw, expectedArgs);
	}

	public static String[] efficientSplitArgs(String args, int expectedArgs) {
		List<String> result = new ArrayList<>();
		boolean inAString = false, escaping = false;
		StringBuilder currentBlock = new StringBuilder();
		char[] array = args.toCharArray();
		for (char c : array) {
			if (escaping) {
				escaping = false;
				currentBlock.append(escape(c));
				continue;
			}

			if (c == '\\') {
				escaping = true;
				continue;
			}

			if (c == '"') {
				inAString = !inAString;
				continue;
			}

			if (inAString) {
				currentBlock.append(c);
				continue;
			}

			if (Character.isSpaceChar(c)) {
				String block = currentBlock.toString();
				currentBlock = new StringBuilder();

				if (!block.isEmpty()) {
					result.add(currentBlock.toString());
					currentBlock = new StringBuilder();
				}
				continue;
			}

			currentBlock.append(c);
		}

		String[] raw = result.toArray(new String[result.size()]);
		if (expectedArgs < 1) return raw;
		return normalizeArray(raw, expectedArgs);
	}

	private static char escape(char c) {
		if (c == 'n') return '\n';
		if (c == 'r') return '\r';
		if (c == 't') return '\t';
		return c;
	}

	public static boolean isNullOrEmpty(String s) {
		return s == null || s.isEmpty();
	}

	public static String limit(String value, int length) {
		StringBuilder buf = new StringBuilder(value);
		if (buf.length() > length) {
			buf.setLength(length - 3);
			buf.append("...");
		}

		return buf.toString();
	}

	public static void main(String[] argv) throws Exception {
		long start = System.currentTimeMillis();
		for (int ii = 0; ii < 1000000; ii++) {
			advancedSplitArgs("a b c d e f g h i j k l m n o p q r s t u v w x y z", 10);
			/*
			10 - elapsed time = 22577ms - 22.577 microseconds per execution
			100 - elapsed time = 21459ms - 21.459 microseconds per execution
			 */

			efficientSplitArgs("a b c d e f g h i j k l m n o p q r s t u v w x y z", 0);
			/*
			0 - elapsed time = 29098ms - 29.098 microseconds per execution
			10 - elapsed time = 22409ms - 22.409 microseconds per execution
			 */

			//splitArgs("a b c d e f g h i j k l m n o p q r s t u v w x y z", 10);
			/*
			10 - elapsed time = 1407ms - 1.407 microseconds per execution
			100 - elapsed time = 2028ms - 2.028 microseconds per execution
			 */

			//SPLIT_PATTERN.split("a b c d e f g h i j k l m n o p q r s t u v w x y z", 10);
			/*
			10 - elapsed time = 1365ms - 1.365 microseconds per execution
			100 - elapsed time = 1851ms -1.851 microseconds per execution
			 */
		}
		long elapsed = System.currentTimeMillis() - start;
		System.out.println("elapsed time = " + elapsed + "ms - " + (elapsed * 1000.0) / 1000000 + " microseconds per execution");
	}

	/**
	 * Normalize an {@link String} Array.
	 *
	 * @param raw          the String array to be normalized
	 * @param expectedSize the final size of the Array.
	 * @return {@link String}[] with the size of expectedArgs
	 */
	public static String[] normalizeArray(String[] raw, int expectedSize) {
		String[] normalized = new String[expectedSize];

		Arrays.fill(normalized, "");
		for (int i = 0; i < normalized.length; i++)
			if (i < raw.length && raw[i] != null && !raw[i].isEmpty()) normalized[i] = raw[i];
		return normalized;
	}

	public static String notNullOrDefault(String str, String defaultStr) {
		if (str == null || str.trim().isEmpty()) return defaultStr;
		return str;
	}

	public static Map<String, String> parse(String[] args) {
		Map<String, String> options = new HashMap<>();

		for (int i = 0; i < args.length; i++) {
			if (args[i].charAt(0) == '-' || args[i].charAt(0) == '/') //This start with - or /
			{
				args[i] = args[i].substring(1);
				if (i + 1 >= args.length || args[i + 1].charAt(0) == '-' || args[i + 1].charAt(0) == '/') //Next start with - (or last arg)
				{
					options.put(args[i], "null");
				} else {
					options.put(args[i], args[i + 1]);
					i++;
				}
			} else {
				options.put(null, args[i]);
			}
		}

		return options;
	}

	public static String removeLines(String str, int startline, int numlines) {
		try (BufferedReader br = new BufferedReader(new StringReader(str))) {
			//String buffer to store contents of the file
			StringBuilder builder = new StringBuilder("");

			//Keep track of the line number
			int linenumber = 0;
			numlines--;
			String line;

			while ((line = br.readLine()) != null) {
				//Store each valid line in the string buffer
				if (linenumber < startline || linenumber >= startline + numlines)
					builder.append(line).append("\n");
				linenumber++;
			}

			return builder.toString();
		} catch (RuntimeException e) {
			throw e;
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * Enchanced {@link String#split(String, int)} with SPLIT_PATTERN as the Pattern used.
	 *
	 * @param args         the {@link String} to be split.
	 * @param expectedArgs the size of the returned array of Non-null {@link String}s
	 * @return a {@link String}[] with the size of expectedArgs
	 */
	public static String[] splitArgs(String args, int expectedArgs) {
		String[] raw = SPLIT_PATTERN.split(args, expectedArgs);
		if (expectedArgs < 1) return raw;
		return normalizeArray(raw, expectedArgs);
	}
}