package br.com.brjdevs.java.utils.collections;

import java.util.*;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class CollectionUtils {
	public static <T, R> List<R> apply(List<T> list, Function<T, R> mapper) {
		return list.stream().map(mapper).collect(Collectors.toList());
	}

	public static Iterable<String> asIterable(Pattern pattern, String string) {
		return () -> {
			Matcher matcher = pattern.matcher(string);
			return new Iterator<String>() {
				@Override
				public boolean hasNext() {
					return matcher.find();
				}

				@Override
				public String next() {
					return matcher.group();
				}
			};
		};
	}

	@SuppressWarnings("unchecked")
	public static <T> Comparator<T> randomOrder() {
		ThreadLocalRandom r = ThreadLocalRandom.current();
		int x = r.nextInt(), y = r.nextInt();
		boolean b = r.nextBoolean();
		return Comparator.comparingInt((T t) -> t.hashCode() ^ x)
			.thenComparingInt(s -> s.toString().length() ^ y)
			.thenComparing(((Comparator<T>)(b ? Comparator.naturalOrder() : Comparator.reverseOrder())));
	}

	public static <T, U> Map<T, U> concatMaps(Map<? extends T, ? extends U> map1, Map<? extends T, ? extends U> map2) {
		return Stream.concat(map1.entrySet().stream(), map2.entrySet().stream())
			.collect(Collectors.toMap(
				Map.Entry::getKey,
				Map.Entry::getValue,
				(entry1, entry2) -> (entry1 == null ? entry2 : entry1)
				)
			);
	}

	public static <T> T random(List<T> list, Random random) {
		return list.get(random.nextInt(list.size()));
	}

	public static <T> T random(T[] array, Random random) {
		return array[random.nextInt(array.length)];
	}

	public static <T> T random(List<T> list) {
		return list.get(random(list.size()));
	}

	public static <T> T random(T[] array) {
		return array[random(array.length)];
	}

	public static int random(int size) {
		return (int) Math.floor(Math.random() * size);
	}

	public static <T> List<T> subListOn(List<T> list, Predicate<T> predicate) {
		Optional<T> first = list.stream().filter(predicate).findFirst();
		return first.map(t -> list.subList(0, list.indexOf(t))).orElse(list);
	}

	public static <T> String toString(Collection<T> collection, Function<T, String> toString, String join) {
		return collection.stream().map(toString).collect(Collectors.joining(join));
	}
}
