package br.com.brjdevs.utils.trove.threads;

import br.com.brjdevs.java.utils.annotations.MissingDocumentation;
import gnu.trove.TCollections;
import gnu.trove.map.TLongObjectMap;
import gnu.trove.map.hash.TLongObjectHashMap;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;

@MissingDocumentation
public class ScheduledTaskProcessor {
	private final Set<Runnable> expirables = Collections.newSetFromMap(new ConcurrentHashMap<>());
	private final TLongObjectMap<Set<Runnable>> expirations = TCollections.synchronizedMap(new TLongObjectHashMap<>());
	private final Consumer<Runnable> onExpired;
	private boolean updated = false;

	public ScheduledTaskProcessor(Consumer<Runnable> onExpired, String name) {
		this.onExpired = onExpired;

		Thread thread = new Thread(this::threadcode, name);
		thread.setDaemon(true);
		thread.start();
	}

	public ScheduledTaskProcessor(String name) {
		this(r -> new Thread(r, "Scheduled Task Thread").start(), name);
	}

	public void addTask(long milis, Runnable task) {
		Objects.requireNonNull(task);

		synchronized (expirations) {
			if (expirables.contains(task)) {
				expirations.valueCollection().forEach(list -> list.remove(task));
				expirables.remove(task);
			}

			if (!expirations.containsKey(milis)) expirations.put(milis, new LinkedHashSet<>());

			expirations.get(milis).add(task);
			expirables.add(task);
		}

		updated = true;
		synchronized (this) {
			notify();
		}
	}

	public void addTaskRelative(long millisToAwait, Runnable task) {
		addTask(System.currentTimeMillis() + millisToAwait, task);
	}

	public void removeTask(Runnable task) {
		Objects.requireNonNull(task);

		synchronized (expirations) {
			if (expirables.contains(task)) {
				expirations.valueCollection().forEach(list -> list.remove(task));
				expirables.remove(task);
			}
		}

		updated = true;
		synchronized (this) {
			notify();
		}
	}

	private void threadcode() {
		//noinspection InfiniteLoopStatement
		while (true) {
			if (expirations.isEmpty()) {
				try {
					synchronized (this) {
						wait();
						updated = false;
					}
				} catch (InterruptedException ignored) {}
			}

			long[] keys = expirations.keys();

			long firstEntry = keys[0];

			//Find Minimal
			for (int i = 1; i < keys.length; i++) if (keys[i] < firstEntry) firstEntry = keys[i];

			long timeout = firstEntry - System.currentTimeMillis();
			if (timeout > 0) {
				synchronized (this) {
					try {
						wait(timeout);
					} catch (InterruptedException ignored) {}
				}
			}

			if (!updated) {
				Set<Runnable> first = expirations.remove(firstEntry);
				first.remove(null);
				first.forEach(onExpired);
			} else updated = false; //and the loop will restart and resolve it
		}
	}

	public OptionalLong timeLeft(Runnable task) {
		synchronized (expirations) {
			OptionalLong first = Arrays.stream(expirations.keys())
				.filter(value -> expirations.get(value).contains(task))
				.findFirst();

			return !first.isPresent() ? first : OptionalLong.of(first.getAsLong() - System.currentTimeMillis());
		}
	}
}