package br.com.brjdevs.java.snowflakes.entities.impl;

import br.com.brjdevs.java.snowflakes.entities.Config;
import br.com.brjdevs.java.snowflakes.entities.Datacenter;
import br.com.brjdevs.java.snowflakes.entities.Worker;
import br.com.brjdevs.java.snowflakes.entities.impl.ConfigImpl.DatacenterImpl.WorkerImpl;

import java.time.OffsetDateTime;
import java.util.Calendar;
import java.util.HashMap;
import java.util.Map;
import java.util.TimeZone;

public class ConfigImpl implements Config {
	public class DatacenterImpl implements Datacenter {
		public class WorkerImpl implements Worker {
			private final long workerId;
			private long lastTimestamp = -1L;
			private long sequence = 0L;

			public WorkerImpl(long workerId) {
				// sanity check for workerId
				if (workerId > maxWorkerId || workerId < 0) {
					throw new IllegalArgumentException(
						String.format("worker Id can't be greater than %d or less than 0", maxWorkerId));
				}

				this.workerId = workerId;
			}

			@Override
			public Datacenter datacenter() {
				return DatacenterImpl.this;
			}

			@Override
			public long generate() {
				long timestamp = timeGen();

				if (timestamp < lastTimestamp) {
					throw new IllegalStateException(String.format(
						"Clock moved backwards. Refusing to generate id for %d milliseconds",
						lastTimestamp - timestamp
					));
				}

				synchronized (this) {
					if (lastTimestamp == timestamp) {
						sequence = (sequence + 1) & sequenceMask;
						if (sequence == 0) timestamp = tilNextMillis(lastTimestamp);
					} else sequence = 0L;

					lastTimestamp = timestamp;

					return ((timestamp - epoch) << timestampLeftShift) | (datacenterId << datacenterIdShift) | (workerId << workerIdShift) | sequence;
				}
			}

			@Override
			public long id() {
				return workerId;
			}

			@Override
			public Config setup() {
				return ConfigImpl.this;
			}

			public long getDatacenterId() {
				return datacenterId;
			}

			public long getWorkerId() {
				return workerId;
			}

			private long tilNextMillis(long lastTimestamp) {
				long timestamp = timeGen();
				while (timestamp <= lastTimestamp) timestamp = timeGen();
				return timestamp;
			}
		}

		private final Map<Long, WorkerImpl> cachedWorkers = new HashMap<>();
		private final long datacenterId;

		private DatacenterImpl(long datacenterId) {
			// sanity check for datacenterId
			if (datacenterId > maxDatacenterId || datacenterId < 0) {
				throw new IllegalArgumentException(
					String.format("datacenter Id can't be greater than %d or less than 0", maxDatacenterId));
			}

			this.datacenterId = datacenterId;
		}

		@Override
		public long id() {
			return datacenterId;
		}

		@Override
		public Config setup() {
			return ConfigImpl.this;
		}

		@Override
		public WorkerImpl worker(long workerId) {
			return cachedWorkers.computeIfAbsent(workerId, WorkerImpl::new);
		}
	}

	private static OffsetDateTime convert(long millis) {
		Calendar gmt = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
		gmt.setTimeInMillis(millis);
		return OffsetDateTime.ofInstant(gmt.toInstant(), gmt.getTimeZone().toZoneId());
	}

	private final Map<Long, DatacenterImpl> cachedDatacenters = new HashMap<>();
	private final long datacenterIdBits;
	private final long datacenterIdShift;
	private final long epoch;
	private final long maxDatacenterId;
	private final long maxWorkerId;
	private final long sequenceBits;
	private final long sequenceMask;
	private final long timestampLeftShift;
	private final long workerIdBits;
	private final long workerIdShift;

	public ConfigImpl(long epoch, long datacenterIdBits, long workerIdBits, long sequenceBits) {
		if (epoch < 0) throw new IllegalArgumentException("epoch must be positive");
		if (epoch > timeGen()) throw new IllegalArgumentException("epoch is on the future");
		if (datacenterIdBits < 0) throw new IllegalArgumentException("datacenterIdBits must be positive");
		if (workerIdBits < 0) throw new IllegalArgumentException("workerIdBits must be positive");
		if (sequenceBits < 0) throw new IllegalArgumentException("sequenceBits must be positive");

		if ((datacenterIdBits + workerIdBits + sequenceBits) >= Long.SIZE)
			throw new IllegalArgumentException(
				"(datacenterIdBits + workerIdBits + sequenceBits) need to be under " + Long.SIZE + " bits.");

		this.datacenterIdBits = datacenterIdBits;
		this.sequenceBits = sequenceBits;
		this.epoch = epoch;
		this.workerIdBits = workerIdBits;

		datacenterIdShift = sequenceBits + workerIdBits;
		timestampLeftShift = datacenterIdShift + datacenterIdBits;
		workerIdShift = this.sequenceBits;
		sequenceMask = ~(-1L << this.sequenceBits);
		maxDatacenterId = ~(-1L << this.datacenterIdBits);
		maxWorkerId = ~(-1L << this.workerIdBits);
	}

	@Override
	public OffsetDateTime creationTime(long snowflake) {
		return convert(creationTimeMillis(snowflake));
	}

	public long creationTimeMillis(long snowflake) {
		return (snowflake >> timestampLeftShift) + epoch;
	}

	@Override
	public DatacenterImpl datacenter(long datacenterId) {
		return cachedDatacenters.computeIfAbsent(datacenterId, DatacenterImpl::new);
	}

	@Override
	public long datacenterBits() {
		return datacenterIdBits;
	}

	@Override
	public long epoch() {
		return epoch;
	}

	@Override
	public OffsetDateTime expirationDate() {
		return convert(expirationDateMillis());
	}

	@Override
	public long expirationDateMillis() {
		return epoch + (2L << (Long.SIZE - timestampLeftShift));
	}

	@Override
	public long sequenceBits() {
		return sequenceBits;
	}

	@Override
	public long timeRemaining() {
		return expirationDateMillis() - System.currentTimeMillis();
	}

	@Override
	public long timestampBits() {
		return Long.SIZE - timestampLeftShift;
	}

	@Override
	public boolean valid() {
		return timeRemaining() >= 0;
	}

	@Override
	public WorkerImpl worker(long datacenterId, long workerId) {
		return datacenter(datacenterId).worker(workerId);
	}

	@Override
	public long workerBits() {
		return workerIdBits;
	}

	public String toString() {
		return String.format(
			"Config[epoch=%d, timestampBits=%d, datacenterIdBits=%d,  workerIdBits=%d, sequenceBits=%d, valid=%s]",
			this.epoch, (Long.SIZE - timestampLeftShift), this.datacenterIdBits, this.workerIdBits, this.sequenceBits,
			valid()
		);
	}

	private void checkValid() {
		if (valid()) throw new IllegalStateException("the config already expired. can't generate new Snowflakes.");
	}

	private long timeGen() {
		return System.currentTimeMillis();
	}
}
