package br.com.brjdevs.network;

import org.java_websocket.WebSocket;
import org.java_websocket.handshake.ClientHandshake;

import java.net.InetSocketAddress;
import java.util.Arrays;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public class Server extends AbstractServer implements Connection {
	private final Map<Integer, WebSocket> connections = new ConcurrentHashMap<>();
	private final SocketListener listener;

	public Server(InetSocketAddress address, PacketRegistry pr, SocketListener listener) {
		super(address, pr);
		this.listener = listener;
		pr.register(JSONPacket.FACTORY);
		pr.lock();
	}

	public Server(InetSocketAddress address, SocketListener listener) {
		this(address, new PacketRegistry(), listener);
	}

	@Override
	public void close(int client) {
		getConnection(client).close();
	}

	@Override
	public WebSocket getSocket(int client) {
		return getConnection(client);
	}

	@Override
	public Connection.Type getType() {
		return Connection.Type.SERVER;
	}

	@Override
	public boolean isConnected(int client) {
		return connections.containsKey(client);
	}

	@Override
	public void reconnect() {
		throw new UnsupportedOperationException();
	}

	@Override
	protected final void connectionOpened(WebSocket socket, ClientHandshake handshake) {
		int id = 0;
		while (connections.containsKey(id)) id++;
		connections.put(id, socket);
		listener.onConnect(this, id, socket);
	}

	@Override
	public final void onPacket(WebSocket sender, Object packet) {
		Object o = listener.onPacket(this, getId(sender), packet);
		if (o == null) return;
		sendPacket(sender, o);
	}

	@Override
	public final void onClose(WebSocket p1, int p2, String p3, boolean p4) {
		int id = getId(p1);
		listener.onClose(this, id, p2, p3);
		connections.remove(id);
	}

	@Override
	public final void onError(WebSocket p1, Exception p2) {
		if (!connections.containsValue(p1))
			listener.onError(this, -1, p2);
		else
			listener.onError(this, getId(p1), p2);
	}

	public final int[] getClients() {
		return Arrays.stream(connections.keySet().toArray(new Integer[0])).mapToInt(Integer::intValue).toArray();
	}

	private WebSocket getConnection(int id) {
		WebSocket s = connections.get(id);
		if (s == null) throw new IllegalArgumentException("No connection for id " + id);
		return s;
	}

	private int getId(WebSocket socket) {
		for (Map.Entry<Integer, WebSocket> entry : connections.entrySet())
			if (entry.getValue().equals(socket)) return entry.getKey();
		throw new IllegalArgumentException("Unknown socket");
	}
}
