package br.com.brjdevs.network;

import br.com.brjdevs.network.factory.PacketFactory;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

public final class PacketRegistry {
	private final Map<Integer, PacketFactory> factories = new HashMap<>();
	private transient boolean lock = false;

	@Override
	public boolean equals(Object other) {
		return other instanceof PacketRegistry && ((PacketRegistry) other).factories.equals(factories);
	}

	@Override
	public String toString() {
		return "PacketRegistry" + factories.toString();
	}

	public void lock() {
		lock = true;
	}

	public Object readPacket(byte[] dataAndId) {
		if (dataAndId.length < 4) throw new IllegalArgumentException("Invalid packet");
		int id = dataAndId[0] << 24 | (dataAndId[1] & 0xFF) << 16 | (dataAndId[2] & 0xFF) << 8 | (dataAndId[3] & 0xFF);
		PacketFactory factory = factories.get(id);
		if (factory == null) throw new IllegalArgumentException("No packet factory found for id " + id);
		return factory.unserialize(Arrays.copyOfRange(dataAndId, 4, dataAndId.length));
	}

	public void register(PacketFactory factory) {
		if (lock) throw new IllegalStateException("Registry locked");
		if (factories.containsKey(factory.id))
			throw new IllegalStateException("PacketFactory for id " + factory.id + " already registered");
		for (PacketFactory f : factories.values())
			if (f.type == factory.type)
				throw new IllegalStateException("PacketFactory for class " + f.type.getName() + " already registered");
		factories.put(factory.id, factory);
	}

	@SuppressWarnings("unchecked")
	public byte[] writePacket(Object packet) {
		Class<?> cls = packet.getClass();
		for (PacketFactory f : factories.values())
			if (f.type == cls) {
				byte[] serialized = f.serialize(packet);
				byte[] ret = new byte[serialized.length + 4];
				ret[0] = (byte) (f.id >> 24);
				ret[1] = (byte) (f.id >> 16);
				ret[2] = (byte) (f.id >> 8);
				ret[3] = (byte) f.id;
				System.arraycopy(serialized, 0, ret, 4, serialized.length);
				return ret;
			}
		throw new IllegalArgumentException("No factory found for class " + cls.getName());
	}
}
