package br.com.brjdevs.network;

import org.java_websocket.WebSocket;
import org.java_websocket.handshake.ServerHandshake;

import java.net.URI;

public class Client implements Connection {
	private static class Socket extends AbstractClient {
		private final Client client;
		private final SocketListener listener;

		public Socket(URI uri, PacketRegistry registry, SocketListener listener, Client client) {
			super(uri, registry, true);
			this.listener = listener;
			this.client = client;
		}

		@Override
		protected void connectionOpened(ServerHandshake handshake) {
			listener.onConnect(client, 0, getConnection());
		}

		@Override
		public void onPacket(Object packet) {
			Object o = listener.onPacket(client, 0, packet);
			if (o == null) return;
			sendPacket(o);
		}

		@Override
		public void onClose(int p1, String p2, boolean p3) {
			listener.onClose(client, 0, p1, p2);
		}

		@Override
		public void onError(Exception p1) {
			listener.onError(client, 0, p1);
		}
	}

	private final SocketListener listener;
	private final PacketRegistry registry;
	private final URI uri;
	private volatile Socket connection;

	public Client(URI uri, PacketRegistry registry, SocketListener listener) {
		this.uri = uri;
		this.registry = registry;
		this.listener = listener;
		registry.register(PacketRegistryAcceptedPacket.FACTORY);
		registry.register(JSONPacket.FACTORY);
		registry.lock();
		connection = new Socket(uri, registry, listener, this);
	}

	public Client(URI uri, SocketListener listener) {
		this(uri, new PacketRegistry(), listener);
	}

	@Override
	public void close(int ignored) {
		try {
			connection.closeBlocking();
		} catch (InterruptedException e) {
		}
	}

	@Override
	public WebSocket getSocket(int client) {
		return connection.getConnection();
	}

	@Override
	public Connection.Type getType() {
		return Connection.Type.CLIENT;
	}

	@Override
	public boolean isConnected(int ignored) {
		return connection.getConnection().isOpen();
	}

	@Override
	public void reconnect() {
		if (connection.getConnection().isOpen())
			try {
				connection.closeBlocking();
			} catch (Exception ignored) {
			}
		connection = new Socket(uri, registry, listener, this);
		try {
			connection.connectBlocking();
			connection.waitForValidation();
		} catch (InterruptedException ignored) {
		}
	}

	public AbstractClient getPacketClient() {
		return connection;
	}
}
