package br.com.brjdevs.network;

import org.java_websocket.WebSocket;
import org.java_websocket.handshake.ClientHandshake;
import org.java_websocket.server.WebSocketServer;

import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

public abstract class AbstractServer extends WebSocketServer {
	private final PacketRegistry registry;
	private final List<WebSocket> socketsWaitingForApproval = new CopyOnWriteArrayList<>();

	public AbstractServer(InetSocketAddress address, PacketRegistry registry) {
		super(address);
		this.registry = registry;
		registry.register(PacketRegistryAcceptedPacket.FACTORY);
	}

	protected abstract void connectionOpened(WebSocket socket, ClientHandshake handshake);

	public abstract void onPacket(WebSocket sender, Object packet);

	@Override
	public final void onOpen(WebSocket p1, ClientHandshake p2) {
		socketsWaitingForApproval.add(p1);
		connectionOpened(p1, p2);
	}

	@Override
	public final void onMessage(WebSocket conn, String message) {
		//not called
	}

	@Override
	public final void onMessage(WebSocket p1, ByteBuffer p2) {
		byte[] data = p2.array();
		if (socketsWaitingForApproval.size() > 0) {
			if (socketsWaitingForApproval.remove(p1)) {
				try {
					PacketRegistryValidationPacket p = PacketRegistryValidationPacket.FACTORY.unserialize(data);
					if (!registry.equals(p.registry)) {
						p1.close(1403, "PacketRegistry does not match");
						return;
					}
				} catch (Exception e) {
					onError(p1, e);
					p1.close(1403, "PacketRegistryValidationPacket invalid");
					return;
				}
				p1.send(registry.writePacket(new PacketRegistryAcceptedPacket()));
			}
		}
		Object packet;
		try {
			packet = registry.readPacket(data);
		} catch (Exception e) {
			onError(p1, e);
			return;
		}
		onPacket(p1, packet);
	}

	public void sendPacket(WebSocket socket, Object packet) {
		socket.send(registry.writePacket(packet));
	}
}
