package br.com.brjdevs.network;

import org.java_websocket.client.WebSocketClient;
import org.java_websocket.handshake.ServerHandshake;

import java.net.URI;
import java.nio.ByteBuffer;

public abstract class AbstractClient extends WebSocketClient {
	private final PacketRegistry registry;
	private volatile boolean closed = false;
	private volatile transient Object lastPacket;
	private volatile boolean validated = false;

	public AbstractClient(URI uri, PacketRegistry registry) {
		this(uri, registry, false);
	}

	public AbstractClient(URI uri, PacketRegistry registry, boolean reusedRegistry) {
		super(uri);
		this.registry = registry;
		if (!reusedRegistry) registry.register(PacketRegistryAcceptedPacket.FACTORY);
	}

	protected abstract void connectionOpened(ServerHandshake handshake);

	public abstract void onPacket(Object packet);

	@Override
	public void onCloseInitiated(int code, String reason) {
		closed = true; //unblock (and throw an exception) any threads waiting for validation if not validated yet
	}

	@Override
	public final void onOpen(ServerHandshake p1) {
		send(PacketRegistryValidationPacket.FACTORY.serialize(new PacketRegistryValidationPacket(registry)));
		connectionOpened(p1);
	}

	@Override
	public final void onMessage(String p1) {
		//not called
	}

	@Override
	public final void onMessage(ByteBuffer bytes) {
		Object packet = registry.readPacket(bytes.array());
		if (packet instanceof PacketRegistryAcceptedPacket) validated = true;
		onPacket(lastPacket = packet);
	}

	public final <T> T readPacketBlocking(Class<T> packetClass) {
		return readPacketBlocking(packetClass, false);
	}

	@SuppressWarnings("unchecked")
	public final <T> T readPacketBlocking(Class<T> packetClass, boolean exact) {
		Object packet = readPacketBlocking();
		if (exact) {
			while (packetClass != packet.getClass()) packet = readPacketBlocking();
			return (T) packet;
		}
		while (!packetClass.isInstance(packet)) packet = readPacketBlocking();
		return (T) packet;
	}

	public final Object readPacketBlocking() {
		Object o = lastPacket;
		while (o == lastPacket) ;
		return lastPacket;
	}

	public final void sendPacket(Object packet) {
		if (!validated) return;
		send(registry.writePacket(packet));
	}

	public final void waitForValidation() {
		while (!validated)
			if (closed)
				throw new IllegalStateException("Connection closed");
	}
}
