/*
 * Copyright 2017 Andrew Rucker Jones.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.anteros.csv.bean.concurrent;

import br.com.anteros.csv.bean.AbstractCSVToBean;
import br.com.anteros.csv.bean.BeanField;
import br.com.anteros.csv.bean.CsvToBeanFilter;
import br.com.anteros.csv.bean.MappingStrategy;
import br.com.anteros.csv.bean.opencsvUtils;
import br.com.anteros.exceptions.CsvBadConverterException;
import br.com.anteros.exceptions.CsvConstraintViolationException;
import br.com.anteros.exceptions.CsvDataTypeMismatchException;
import br.com.anteros.exceptions.CsvException;
import br.com.anteros.exceptions.CsvRequiredFieldEmptyException;

import java.beans.IntrospectionException;
import java.beans.PropertyDescriptor;
import java.beans.PropertyEditor;
import java.lang.reflect.InvocationTargetException;
import java.util.concurrent.BlockingQueue;

/**
 * A class that encapsulates the job of creating a bean from a line of CSV input
 * and making it possible to run those jobs in parallel.
 * @param <T> The type of the bean being created
 * @author Andrew Rucker Jones
 * @since 4.0
 */
public class ProcessCsvLine<T> extends AbstractCSVToBean implements Runnable {
    private final long lineNumber;
    private final MappingStrategy<T> mapper;
    private final CsvToBeanFilter filter;
    private final String[] line;
    private final BlockingQueue<OrderedObject<T>> resultantBeanQueue;
    private final BlockingQueue<OrderedObject<CsvException>> thrownExceptionsQueue;
    private final boolean throwExceptions;
    T bean;

    /**
     * The only constructor for creating a bean out of a line of input.
     * @param lineNumber Which record in the input file is being processed
     * @param mapper The mapping strategy to be used
     * @param filter A filter to remove beans from the running, if necessary.
     *   May be null.
     * @param line The line of input to be transformed into a bean
     * @param resultantBeanQueue A queue in which to place the bean created
     * @param thrownExceptionsQueue A queue in which to place a thrown
     *   exception, if one is thrown
     * @param throwExceptions Whether exceptions should be thrown, ending
     *   processing, or suppressed and saved for later processing
     */
    public ProcessCsvLine(
            long lineNumber, MappingStrategy<T> mapper, CsvToBeanFilter filter,
            String[] line, BlockingQueue<OrderedObject<T>> resultantBeanQueue,
            BlockingQueue<OrderedObject<CsvException>> thrownExceptionsQueue,
            boolean throwExceptions) {
        this.lineNumber = lineNumber;
        this.mapper = mapper;
        this.filter = filter;
        this.line = line;
        this.resultantBeanQueue = resultantBeanQueue;
        this.thrownExceptionsQueue = thrownExceptionsQueue;
        this.throwExceptions = throwExceptions;
    }

    @Override
    public void run() {
        try {
            if (filter == null || filter.allowLine(line)) {
                T obj = processLine();
                opencsvUtils.queueRefuseToAcceptDefeat(
                        resultantBeanQueue,
                        new OrderedObject(lineNumber, obj));
            }
        } catch (CsvException e) {
            CsvException csve = (CsvException) e;
            csve.setLineNumber(lineNumber);
            if (throwExceptions) {
                throw new RuntimeException(csve);
            }
            opencsvUtils.queueRefuseToAcceptDefeat(thrownExceptionsQueue,
                    new OrderedObject(lineNumber, csve));
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Creates a single object from a line from the CSV file.
     * @return Object containing the values.
     * @throws IllegalAccessException Thrown on error creating bean.
     * @throws InvocationTargetException Thrown on error calling the setters.
     * @throws InstantiationException Thrown on error creating bean.
     * @throws IntrospectionException Thrown on error getting the
     *   PropertyDescriptor.
     * @throws CsvBadConverterException If a custom converter cannot be
     *   initialized properly
     * @throws CsvDataTypeMismatchException If the source data cannot be
     *   converted to the type of the destination field
     * @throws CsvRequiredFieldEmptyException If a mandatory field is empty in
     *   the input file
     * @throws CsvConstraintViolationException When the internal structure of
     *   data would be violated by the data in the CSV file
     */
    private T processLine()
            throws IllegalAccessException, InvocationTargetException,
            InstantiationException, IntrospectionException,
            CsvBadConverterException, CsvDataTypeMismatchException,
            CsvRequiredFieldEmptyException, CsvConstraintViolationException {
        mapper.verifyLineLength(line.length);
        bean = mapper.createBean();
        for (int col = 0; col < line.length; col++) {
            if (mapper.isAnnotationDriven()) {
                processField(col);
            } else {
                processProperty(col);
            }
        }
        return bean;
    }

    private void processField(int col)
            throws CsvBadConverterException, CsvDataTypeMismatchException,
            CsvRequiredFieldEmptyException, CsvConstraintViolationException {
        BeanField beanField = mapper.findField(col);
        if (beanField != null) {
            String value = line[col];
            beanField.setFieldValue(bean, value);
        }
    }

    private void processProperty(int col)
            throws IntrospectionException, InstantiationException,
            IllegalAccessException, InvocationTargetException, CsvBadConverterException {
        PropertyDescriptor prop = mapper.findDescriptor(col);
        if (null != prop) {
            String value = checkForTrim(line[col], prop);
            Object obj = convertValue(value, prop);
            prop.getWriteMethod().invoke(bean, obj);
        }
    }

    @Override
    protected PropertyEditor getPropertyEditor(PropertyDescriptor desc) throws InstantiationException, IllegalAccessException {
        Class<?> cls = desc.getPropertyEditorClass();
        if (null != cls) {
            return (PropertyEditor) cls.newInstance();
        }
        return getPropertyEditorValue(desc.getPropertyType());
    }
}
