package biz.k11i.util;

/**
 * Faster implementation of math functions.
 * <p>
 * This implementation includes the part of the works licensed under the Apache License:
 * <ul>
 * <li>
 * commons-math3
 * <ul>
 * <li>{@link #log(double)}</li>
 * </ul>
 * </li>
 * <li>
 * Jafama
 * <ul>
 * <li>{@link #exp(double)}</li>
 * <li>{@link #expm1(double)}</li>
 * <li>{@link #log1p(double)}</li>
 * <li>{@link #pow(double, double)}</li>
 * </ul>
 * </li>
 * </ul>
 */
public final class MathFunctions {
    public static final double log(double value) {
        return CommonsMath.log(value);
    }

    public static final double log1p(double value) {
        return JafamaMath.log1p(value);
    }

    public static final double exp(double value) {
        return JafamaMath.exp(value);
    }

    public static final double expm1(double value) {
        return JafamaMath.expm1(value);
    }

    public static final double pow(double value, double power) {
        return JafamaMath.pow(value, power);
    }

    /*
     * Licensed to the Apache Software Foundation (ASF) under one or more
     * contributor license agreements.  See the NOTICE file distributed with
     * this work for additional information regarding copyright ownership.
     * The ASF licenses this file to You under the Apache License, Version 2.0
     * (the "License"); you may not use this file except in compliance with
     * the License.  You may obtain a copy of the License at
     *
     *      http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     */

    /**
     * Math functions from commons-math3 (3.6)
     */
    private static final class CommonsMath {
        /**
         * Enclose large data table in nested static class so it's only loaded on first access.
         */
        private static class lnMant {
            /**
             * Extended precision logarithm table over the range 1 - 2 in increments of 2^-10.
             */
            private static final double[][] LN_MANT = new double[][]{
                    {+0.0d, +0.0d,}, // 0
                    {+9.760860120877624E-4d, -3.903230345984362E-11d,}, // 1
                    {+0.0019512202125042677d, -8.124251825289188E-11d,}, // 2
                    {+0.0029254043474793434d, -1.8374207360194882E-11d,}, // 3
                    {+0.0038986406289041042d, -2.1324678121885073E-10d,}, // 4
                    {+0.004870930686593056d, -4.5199654318611534E-10d,}, // 5
                    {+0.00584227591753006d, -2.933016992001806E-10d,}, // 6
                    {+0.006812678650021553d, -2.325147219074669E-10d,}, // 7
                    {+0.007782140746712685d, -3.046577356838847E-10d,}, // 8
                    {+0.008750664070248604d, -5.500631513861575E-10d,}, // 9
                    {+0.00971824862062931d, +8.48292035519895E-10d,}, // 10
                    {+0.010684899985790253d, +1.1422610134013436E-10d,}, // 11
                    {+0.01165061630308628d, +9.168889933128375E-10d,}, // 12
                    {+0.012615403160452843d, -5.303786078838E-10d,}, // 13
                    {+0.013579258695244789d, -5.688639355498786E-10d,}, // 14
                    {+0.01454218477010727d, +7.296670293275653E-10d,}, // 15
                    {+0.015504186972975731d, -4.370104767451421E-10d,}, // 16
                    {+0.016465261578559875d, +1.43695591408832E-9d,}, // 17
                    {+0.01742541790008545d, -1.1862263158849434E-9d,}, // 18
                    {+0.018384650349617004d, -9.482976524690715E-10d,}, // 19
                    {+0.01934296265244484d, +1.9068609515836638E-10d,}, // 20
                    {+0.020300358533859253d, +2.655990315697216E-10d,}, // 21
                    {+0.021256837993860245d, +1.0315548713040775E-9d,}, // 22
                    {+0.022212404757738113d, +5.13345647019085E-10d,}, // 23
                    {+0.02316705882549286d, +4.5604151934208014E-10d,}, // 24
                    {+0.02412080392241478d, -1.1255706987475148E-9d,}, // 25
                    {+0.025073636323213577d, +1.2289023836765196E-9d,}, // 26
                    {+0.02602556347846985d, +1.7990281828096504E-9d,}, // 27
                    {+0.026976589113473892d, -1.4152718164638451E-9d,}, // 28
                    {+0.02792670577764511d, +7.568772963781632E-10d,}, // 29
                    {+0.0288759246468544d, -1.1449998592111558E-9d,}, // 30
                    {+0.029824241995811462d, -1.6850976862319495E-9d,}, // 31
                    {+0.030771657824516296d, +8.422373919843096E-10d,}, // 32
                    {+0.0317181795835495d, +6.872350402175489E-10d,}, // 33
                    {+0.03266380727291107d, -4.541194749189272E-10d,}, // 34
                    {+0.03360854089260101d, -8.9064764856495E-10d,}, // 35
                    {+0.034552380442619324d, +1.0640404096769032E-9d,}, // 36
                    {+0.0354953333735466d, -3.5901655945224663E-10d,}, // 37
                    {+0.03643739968538284d, -3.4829517943661266E-9d,}, // 38
                    {+0.037378571927547455d, +8.149473794244232E-10d,}, // 39
                    {+0.03831886500120163d, -6.990650304449166E-10d,}, // 40
                    {+0.03925827145576477d, +1.0883076226453258E-9d,}, // 41
                    {+0.040196798741817474d, +3.845192807999274E-10d,}, // 42
                    {+0.04113444685935974d, -1.1570594692045927E-9d,}, // 43
                    {+0.04207121580839157d, -1.8877045166697178E-9d,}, // 44
                    {+0.043007105588912964d, -1.6332083257987747E-10d,}, // 45
                    {+0.04394212365150452d, -1.7950057534514933E-9d,}, // 46
                    {+0.04487626254558563d, +2.302710041648838E-9d,}, // 47
                    {+0.045809537172317505d, -1.1410233017161343E-9d,}, // 48
                    {+0.04674194008111954d, -3.0498741599744685E-9d,}, // 49
                    {+0.04767347127199173d, -1.8026348269183678E-9d,}, // 50
                    {+0.04860413819551468d, -3.233204600453039E-9d,}, // 51
                    {+0.04953393340110779d, +1.7211688427961583E-9d,}, // 52
                    {+0.05046287178993225d, -2.329967807055457E-10d,}, // 53
                    {+0.05139094591140747d, -4.191810118556531E-11d,}, // 54
                    {+0.052318163216114044d, -3.5574324788328143E-9d,}, // 55
                    {+0.053244516253471375d, -1.7346590916458485E-9d,}, // 56
                    {+0.05417001247406006d, -4.343048751383674E-10d,}, // 57
                    {+0.055094651877880096d, +1.92909364037955E-9d,}, // 58
                    {+0.056018441915512085d, -5.139745677199588E-10d,}, // 59
                    {+0.05694137513637543d, +1.2637629975129189E-9d,}, // 60
                    {+0.05786345899105072d, +1.3840561112481119E-9d,}, // 61
                    {+0.058784693479537964d, +1.414889689612056E-9d,}, // 62
                    {+0.05970507860183716d, +2.9199191907666474E-9d,}, // 63
                    {+0.0606246218085289d, +7.90594243412116E-12d,}, // 64
                    {+0.06154331564903259d, +1.6844747839686189E-9d,}, // 65
                    {+0.06246116757392883d, +2.0498074572151747E-9d,}, // 66
                    {+0.06337818503379822d, -4.800180493433863E-9d,}, // 67
                    {+0.06429435312747955d, -2.4220822960064277E-9d,}, // 68
                    {+0.06520968675613403d, -4.179048566709334E-9d,}, // 69
                    {+0.06612417101860046d, +6.363872957010456E-9d,}, // 70
                    {+0.06703783571720123d, +9.339468680056365E-10d,}, // 71
                    {+0.06795066595077515d, -4.04226739708981E-9d,}, // 72
                    {+0.0688626617193222d, -7.043545052852817E-9d,}, // 73
                    {+0.06977382302284241d, -6.552819560439773E-9d,}, // 74
                    {+0.07068414986133575d, -1.0571674860370546E-9d,}, // 75
                    {+0.07159365713596344d, -3.948954622015801E-9d,}, // 76
                    {+0.07250232994556427d, +1.1776625988228244E-9d,}, // 77
                    {+0.07341018319129944d, +9.221072639606492E-10d,}, // 78
                    {+0.07431721687316895d, -3.219119568928366E-9d,}, // 79
                    {+0.0752234160900116d, +5.147575929018918E-9d,}, // 80
                    {+0.07612881064414978d, -2.291749683541979E-9d,}, // 81
                    {+0.07703337073326111d, +5.749565906124772E-9d,}, // 82
                    {+0.07793712615966797d, +9.495158151301779E-10d,}, // 83
                    {+0.07884006202220917d, -3.144331429489291E-10d,}, // 84
                    {+0.0797421783208847d, +3.430029236134205E-9d,}, // 85
                    {+0.08064348995685577d, -1.2499290483167703E-9d,}, // 86
                    {+0.08154398202896118d, +2.011215719133196E-9d,}, // 87
                    {+0.08244366943836212d, -2.2728753031387152E-10d,}, // 88
                    {+0.0833425521850586d, -6.508966857277253E-9d,}, // 89
                    {+0.0842406153678894d, -4.801131671405377E-10d,}, // 90
                    {+0.08513787388801575d, +4.406750291994231E-9d,}, // 91
                    {+0.08603434264659882d, -5.304795662536171E-9d,}, // 92
                    {+0.08692999184131622d, +1.6284313912612293E-9d,}, // 93
                    {+0.08782485127449036d, -3.158898981674071E-9d,}, // 94
                    {+0.08871890604496002d, -3.3324878834139977E-9d,}, // 95
                    {+0.08961215615272522d, +2.536961912893389E-9d,}, // 96
                    {+0.09050461649894714d, +9.737596728980696E-10d,}, // 97
                    {+0.0913962870836258d, -6.600437262505396E-9d,}, // 98
                    {+0.09228715300559998d, -3.866609889222889E-9d,}, // 99
                    {+0.09317722916603088d, -4.311847594020281E-9d,}, // 100
                    {+0.09406651556491852d, -6.525851105645959E-9d,}, // 101
                    {+0.09495499730110168d, +5.799080912675435E-9d,}, // 102
                    {+0.09584270417690277d, +4.2634204358490415E-9d,}, // 103
                    {+0.09672962129116058d, +5.167390528799477E-9d,}, // 104
                    {+0.09761576354503632d, -4.994827392841906E-9d,}, // 105
                    {+0.09850110113620758d, +4.970725577861395E-9d,}, // 106
                    {+0.09938566386699677d, +6.6496705953229645E-9d,}, // 107
                    {+0.10026945173740387d, +1.4262712796792241E-9d,}, // 108
                    {+0.1011524498462677d, +5.5822855204629114E-9d,}, // 109
                    {+0.10203467309474945d, +5.593494835247651E-9d,}, // 110
                    {+0.10291612148284912d, +2.8332008343480686E-9d,}, // 111
                    {+0.10379679501056671d, -1.3289231465997192E-9d,}, // 112
                    {+0.10467669367790222d, -5.526819276639527E-9d,}, // 113
                    {+0.10555580258369446d, +6.503128678219282E-9d,}, // 114
                    {+0.10643415153026581d, +6.317463237641817E-9d,}, // 115
                    {+0.10731174051761627d, -4.728528221305482E-9d,}, // 116
                    {+0.10818853974342346d, +4.519199083083901E-9d,}, // 117
                    {+0.10906457901000977d, +5.606492666349878E-9d,}, // 118
                    {+0.10993985831737518d, -1.220176214398581E-10d,}, // 119
                    {+0.11081436276435852d, +3.5759315936869937E-9d,}, // 120
                    {+0.11168810725212097d, +3.1367659571899855E-9d,}, // 121
                    {+0.11256109178066254d, -1.0543075713098835E-10d,}, // 122
                    {+0.11343331634998322d, -4.820065619207094E-9d,}, // 123
                    {+0.11430476605892181d, +5.221136819669415E-9d,}, // 124
                    {+0.11517547070980072d, +1.5395018670011342E-9d,}, // 125
                    {+0.11604541540145874d, +3.5638391501880846E-10d,}, // 126
                    {+0.11691460013389587d, +2.9885336757136527E-9d,}, // 127
                    {+0.11778303980827332d, -4.151889860890893E-9d,}, // 128
                    {+0.11865071952342987d, -4.853823938804204E-9d,}, // 129
                    {+0.11951763927936554d, +2.189226237170704E-9d,}, // 130
                    {+0.12038381397724152d, +3.3791993048776982E-9d,}, // 131
                    {+0.1212492436170578d, +1.5811884868243975E-11d,}, // 132
                    {+0.12211392819881439d, -6.6045909118908625E-9d,}, // 133
                    {+0.1229778528213501d, -2.8786263916116364E-10d,}, // 134
                    {+0.12384103238582611d, +5.354472503748251E-9d,}, // 135
                    {+0.12470348179340363d, -3.2924463896248744E-9d,}, // 136
                    {+0.12556517124176025d, +4.856678149580005E-9d,}, // 137
                    {+0.12642613053321838d, +1.2791850600366742E-9d,}, // 138
                    {+0.12728634476661682d, +2.1525945093362843E-9d,}, // 139
                    {+0.12814581394195557d, +8.749974471767862E-9d,}, // 140
                    {+0.129004567861557d, -7.461209161105275E-9d,}, // 141
                    {+0.12986254692077637d, +1.4390208226263824E-8d,}, // 142
                    {+0.1307198405265808d, -1.3839477920475328E-8d,}, // 143
                    {+0.13157635927200317d, -1.483283901239408E-9d,}, // 144
                    {+0.13243216276168823d, -6.889072914229094E-9d,}, // 145
                    {+0.1332872211933136d, +9.990351100568362E-10d,}, // 146
                    {+0.13414156436920166d, -6.370937412495338E-9d,}, // 147
                    {+0.13499516248703003d, +2.05047480130511E-9d,}, // 148
                    {+0.1358480453491211d, -2.29509872547079E-9d,}, // 149
                    {+0.13670018315315247d, +1.16354361977249E-8d,}, // 150
                    {+0.13755163550376892d, -1.452496267904829E-8d,}, // 151
                    {+0.1384023129940033d, +9.865115839786888E-9d,}, // 152
                    {+0.13925230503082275d, -3.369999130712228E-9d,}, // 153
                    {+0.14010155200958252d, +6.602496401651853E-9d,}, // 154
                    {+0.14095008373260498d, +1.1205312852298845E-8d,}, // 155
                    {+0.14179790019989014d, +1.1660367213160203E-8d,}, // 156
                    {+0.142645001411438d, +9.186471222585239E-9d,}, // 157
                    {+0.14349138736724854d, +4.999341878263704E-9d,}, // 158
                    {+0.14433705806732178d, +3.11611905696257E-10d,}, // 159
                    {+0.14518201351165771d, -3.6671598175618173E-9d,}, // 160
                    {+0.14602625370025635d, -5.730477881659618E-9d,}, // 161
                    {+0.14686977863311768d, -4.674900007989718E-9d,}, // 162
                    {+0.1477125883102417d, +6.999732437141968E-10d,}, // 163
                    {+0.14855468273162842d, +1.159150872494107E-8d,}, // 164
                    {+0.14939609169960022d, -6.082714828488485E-10d,}, // 165
                    {+0.15023678541183472d, -4.905712741596318E-9d,}, // 166
                    {+0.1510767638683319d, -1.124848988733307E-10d,}, // 167
                    {+0.15191605687141418d, -1.484557220949851E-8d,}, // 168
                    {+0.15275460481643677d, +1.1682026251371384E-8d,}, // 169
                    {+0.15359249711036682d, -8.757272519238786E-9d,}, // 170
                    {+0.15442964434623718d, +1.4419920764774415E-8d,}, // 171
                    {+0.15526613593101501d, -7.019891063126053E-9d,}, // 172
                    {+0.15610191226005554d, -1.230153548825964E-8d,}, // 173
                    {+0.15693697333335876d, -2.574172005933276E-10d,}, // 174
                    {+0.15777134895324707d, +4.748140799544371E-10d,}, // 175
                    {+0.15860503911972046d, -8.943081874891003E-9d,}, // 176
                    {+0.15943801403045654d, +2.4500739038517657E-9d,}, // 177
                    {+0.1602703034877777d, +6.007922084557054E-9d,}, // 178
                    {+0.16110190749168396d, +2.8835418231126645E-9d,}, // 179
                    {+0.1619328260421753d, -5.772862039728412E-9d,}, // 180
                    {+0.16276302933692932d, +1.0988372954605789E-8d,}, // 181
                    {+0.16359257698059082d, -5.292913162607026E-9d,}, // 182
                    {+0.16442140936851501d, +6.12956339275823E-9d,}, // 183
                    {+0.16524958610534668d, -1.3210039516811888E-8d,}, // 184
                    {+0.16607704758644104d, -2.5711014608334873E-9d,}, // 185
                    {+0.16690382361412048d, +9.37721319457112E-9d,}, // 186
                    {+0.1677299439907074d, -6.0370682395944045E-9d,}, // 187
                    {+0.168555349111557d, +1.1918249660105651E-8d,}, // 188
                    {+0.1693800985813141d, +4.763282949656017E-9d,}, // 189
                    {+0.17020416259765625d, +3.4223342273948817E-9d,}, // 190
                    {+0.1710275411605835d, +9.014612241310916E-9d,}, // 191
                    {+0.1718502640724182d, -7.145758990550526E-9d,}, // 192
                    {+0.172672301530838d, -1.4142763934081504E-8d,}, // 193
                    {+0.1734936535358429d, -1.0865453656579032E-8d,}, // 194
                    {+0.17431432008743286d, +3.794385569450774E-9d,}, // 195
                    {+0.1751343309879303d, +1.1399188501627291E-9d,}, // 196
                    {+0.17595365643501282d, +1.2076238768270153E-8d,}, // 197
                    {+0.1767723262310028d, +7.901084730502162E-9d,}, // 198
                    {+0.17759034037590027d, -1.0288181007465474E-8d,}, // 199
                    {+0.1784076690673828d, -1.15945645153806E-8d,}, // 200
                    {+0.17922431230545044d, +5.073923825786778E-9d,}, // 201
                    {+0.18004029989242554d, +1.1004278077575267E-8d,}, // 202
                    {+0.1808556318283081d, +7.2831502374676964E-9d,}, // 203
                    {+0.18167030811309814d, -5.0054634662706464E-9d,}, // 204
                    {+0.18248429894447327d, +5.022108460298934E-9d,}, // 205
                    {+0.18329763412475586d, +8.642254225732676E-9d,}, // 206
                    {+0.18411031365394592d, +6.931054493326395E-9d,}, // 207
                    {+0.18492233753204346d, +9.619685356326533E-10d,}, // 208
                    {+0.18573370575904846d, -8.194157257980706E-9d,}, // 209
                    {+0.18654438853263855d, +1.0333241479437797E-8d,}, // 210
                    {+0.1873544454574585d, -1.9948340196027965E-9d,}, // 211
                    {+0.1881638467311859d, -1.4313002926259948E-8d,}, // 212
                    {+0.1889725625514984d, +4.241536392174967E-9d,}, // 213
                    {+0.18978065252304077d, -4.877952454011428E-9d,}, // 214
                    {+0.1905880868434906d, -1.0813801247641613E-8d,}, // 215
                    {+0.1913948655128479d, -1.2513218445781325E-8d,}, // 216
                    {+0.19220098853111267d, -8.925958555729115E-9d,}, // 217
                    {+0.1930064558982849d, +9.956860681280245E-10d,}, // 218
                    {+0.193811297416687d, -1.1505428993246996E-8d,}, // 219
                    {+0.1946154534816742d, +1.4217997464522202E-8d,}, // 220
                    {+0.19541901350021362d, -1.0200858727747717E-8d,}, // 221
                    {+0.19622188806533813d, +5.682607223902455E-9d,}, // 222
                    {+0.1970241367816925d, +3.2988908516009827E-9d,}, // 223
                    {+0.19782572984695435d, +1.3482965534659446E-8d,}, // 224
                    {+0.19862669706344604d, +7.462678536479685E-9d,}, // 225
                    {+0.1994270384311676d, -1.3734273888891115E-8d,}, // 226
                    {+0.20022669434547424d, +1.0521983802642893E-8d,}, // 227
                    {+0.20102575421333313d, -8.152742388541905E-9d,}, // 228
                    {+0.2018241584300995d, -9.133484280193855E-9d,}, // 229
                    {+0.20262190699577332d, +8.59763959528144E-9d,}, // 230
                    {+0.2034190595149994d, -1.3548568223001414E-8d,}, // 231
                    {+0.20421552658081055d, +1.4847880344628818E-8d,}, // 232
                    {+0.20501139760017395d, +5.390620378060543E-9d,}, // 233
                    {+0.2058066427707672d, -1.1109834472051523E-8d,}, // 234
                    {+0.20660123229026794d, -3.845373872038116E-9d,}, // 235
                    {+0.20739519596099854d, -1.6149279479975042E-9d,}, // 236
                    {+0.20818853378295898d, -3.4174925203771133E-9d,}, // 237
                    {+0.2089812457561493d, -8.254443919468538E-9d,}, // 238
                    {+0.20977330207824707d, +1.4672790944499144E-8d,}, // 239
                    {+0.2105647623538971d, +6.753452542942992E-9d,}, // 240
                    {+0.21135559678077698d, -1.218609462241927E-9d,}, // 241
                    {+0.21214580535888672d, -8.254218316367887E-9d,}, // 242
                    {+0.21293538808822632d, -1.3366540360587255E-8d,}, // 243
                    {+0.2137243151664734d, +1.4231244750190031E-8d,}, // 244
                    {+0.2145126760005951d, -1.3885660525939072E-8d,}, // 245
                    {+0.21530038118362427d, -7.3304404046850136E-9d,}, // 246
                    {+0.2160874605178833d, +5.072117654842356E-9d,}, // 247
                    {+0.21687394380569458d, -5.505080220459036E-9d,}, // 248
                    {+0.21765980124473572d, -8.286782292266659E-9d,}, // 249
                    {+0.2184450328350067d, -2.302351152358085E-9d,}, // 250
                    {+0.21922963857650757d, +1.3416565858314603E-8d,}, // 251
                    {+0.22001364827156067d, +1.0033721426962048E-8d,}, // 252
                    {+0.22079706192016602d, -1.1487079818684332E-8d,}, // 253
                    {+0.22157981991767883d, +9.420348186357043E-9d,}, // 254
                    {+0.2223619818687439d, +1.4110645699377834E-8d,}, // 255
                    {+0.2231435477733612d, +3.5408485497116107E-9d,}, // 256
                    {+0.22392448782920837d, +8.468072777056227E-9d,}, // 257
                    {+0.2247048318386078d, +4.255446699237779E-11d,}, // 258
                    {+0.22548454999923706d, +9.016946273084244E-9d,}, // 259
                    {+0.22626367211341858d, +6.537034810260226E-9d,}, // 260
                    {+0.22704219818115234d, -6.451285264969768E-9d,}, // 261
                    {+0.22782009840011597d, +7.979956357126066E-10d,}, // 262
                    {+0.22859740257263184d, -5.759582672039005E-10d,}, // 263
                    {+0.22937411069869995d, -9.633854121180397E-9d,}, // 264
                    {+0.23015019297599792d, +4.363736368635843E-9d,}, // 265
                    {+0.23092567920684814d, +1.2549416560182509E-8d,}, // 266
                    {+0.231700599193573d, -1.3946383592553814E-8d,}, // 267
                    {+0.2324748933315277d, -1.458843364504023E-8d,}, // 268
                    {+0.23324856162071228d, +1.1551692104697154E-8d,}, // 269
                    {+0.23402166366577148d, +5.795621295524984E-9d,}, // 270
                    {+0.23479416966438293d, -1.1301979046684263E-9d,}, // 271
                    {+0.23556607961654663d, -8.303779721781787E-9d,}, // 272
                    {+0.23633739352226257d, -1.4805271785394075E-8d,}, // 273
                    {+0.23710808157920837d, +1.0085373835899469E-8d,}, // 274
                    {+0.2378782033920288d, +7.679117635349454E-9d,}, // 275
                    {+0.2386477291584015d, +8.69177352065934E-9d,}, // 276
                    {+0.23941665887832642d, +1.4034725764547136E-8d,}, // 277
                    {+0.24018502235412598d, -5.185064518887831E-9d,}, // 278
                    {+0.2409527599811554d, +1.1544236628121676E-8d,}, // 279
                    {+0.24171993136405945d, +5.523085719902123E-9d,}, // 280
                    {+0.24248650670051575d, +7.456824943331887E-9d,}, // 281
                    {+0.24325251579284668d, -1.1555923403029638E-8d,}, // 282
                    {+0.24401789903640747d, +8.988361382732908E-9d,}, // 283
                    {+0.2447827160358429d, +1.0381848020926893E-8d,}, // 284
                    {+0.24554696679115295d, -6.480706118857055E-9d,}, // 285
                    {+0.24631062150001526d, -1.0904271124793968E-8d,}, // 286
                    {+0.2470736801624298d, -1.998183061531611E-9d,}, // 287
                    {+0.247836172580719d, -8.676137737360023E-9d,}, // 288
                    {+0.24859806895256042d, -2.4921733203932487E-10d,}, // 289
                    {+0.2493593990802765d, -5.635173762130303E-9d,}, // 290
                    {+0.2501201629638672d, -2.3951455355985637E-8d,}, // 291
                    {+0.25088030099868774d, +5.287121672447825E-9d,}, // 292
                    {+0.2516399025917053d, -6.447877375049486E-9d,}, // 293
                    {+0.25239890813827515d, +1.32472428796441E-9d,}, // 294
                    {+0.2531573176383972d, +2.9479464287605006E-8d,}, // 295
                    {+0.2539151906967163d, +1.9284247135543574E-8d,}, // 296
                    {+0.2546725273132324d, -2.8390360197221716E-8d,}, // 297
                    {+0.255429208278656d, +6.533522495226226E-9d,}, // 298
                    {+0.2561853528022766d, +5.713225978895991E-9d,}, // 299
                    {+0.25694090127944946d, +2.9618050962556135E-8d,}, // 300
                    {+0.25769591331481934d, +1.950605015323617E-8d,}, // 301
                    {+0.25845038890838623d, -2.3762031507525576E-8d,}, // 302
                    {+0.2592042088508606d, +1.98818938195077E-8d,}, // 303
                    {+0.25995755195617676d, -2.751925069084042E-8d,}, // 304
                    {+0.2607102394104004d, +1.3703391844683932E-8d,}, // 305
                    {+0.26146239042282104d, +2.5193525310038174E-8d,}, // 306
                    {+0.2622140049934387d, +7.802219817310385E-9d,}, // 307
                    {+0.26296502351760864d, +2.1983272709242607E-8d,}, // 308
                    {+0.2637155055999756d, +8.979279989292184E-9d,}, // 309
                    {+0.2644653916358948d, +2.9240221157844312E-8d,}, // 310
                    {+0.265214741230011d, +2.4004885823813374E-8d,}, // 311
                    {+0.2659635543823242d, -5.885186277410878E-9d,}, // 312
                    {+0.2667117714881897d, +1.4300386517357162E-11d,}, // 313
                    {+0.2674594521522522d, -1.7063531531989365E-8d,}, // 314
                    {+0.26820653676986694d, +3.3218524692903896E-9d,}, // 315
                    {+0.2689530849456787d, +2.3998252479954764E-9d,}, // 316
                    {+0.2696990966796875d, -1.8997462070389404E-8d,}, // 317
                    {+0.27044451236724854d, -4.350745270980051E-10d,}, // 318
                    {+0.2711893916130066d, -6.892221115467135E-10d,}, // 319
                    {+0.27193373441696167d, -1.89333199110902E-8d,}, // 320
                    {+0.272677481174469d, +5.262017392507765E-9d,}, // 321
                    {+0.27342069149017334d, +1.3115046679980076E-8d,}, // 322
                    {+0.2741633653640747d, +5.4468361834451975E-9d,}, // 323
                    {+0.2749055027961731d, -1.692337384653611E-8d,}, // 324
                    {+0.27564704418182373d, +6.426479056697412E-9d,}, // 325
                    {+0.2763880491256714d, +1.670735065191342E-8d,}, // 326
                    {+0.27712851762771606d, +1.4733029698334834E-8d,}, // 327
                    {+0.27786844968795776d, +1.315498542514467E-9d,}, // 328
                    {+0.2786078453063965d, -2.2735061539223372E-8d,}, // 329
                    {+0.27934664487838745d, +2.994379757313727E-9d,}, // 330
                    {+0.28008490800857544d, +1.970577274107218E-8d,}, // 331
                    {+0.28082263469696045d, +2.820392733542077E-8d,}, // 332
                    {+0.2815598249435425d, +2.929187356678173E-8d,}, // 333
                    {+0.28229647874832153d, +2.377086680926386E-8d,}, // 334
                    {+0.2830325961112976d, +1.2440393009992529E-8d,}, // 335
                    {+0.2837681770324707d, -3.901826104778096E-9d,}, // 336
                    {+0.2845032215118408d, -2.4459827842685974E-8d,}, // 337
                    {+0.2852376699447632d, +1.1165241398059789E-8d,}, // 338
                    {+0.28597164154052734d, -1.54434478239181E-8d,}, // 339
                    {+0.28670501708984375d, +1.5714110564653245E-8d,}, // 340
                    {+0.28743791580200195d, -1.3782394940142479E-8d,}, // 341
                    {+0.2881702184677124d, +1.6063569876284005E-8d,}, // 342
                    {+0.28890204429626465d, -1.317176818216125E-8d,}, // 343
                    {+0.28963327407836914d, +1.8504673536253893E-8d,}, // 344
                    {+0.29036402702331543d, -7.334319635123628E-9d,}, // 345
                    {+0.29109418392181396d, +2.9300903540317107E-8d,}, // 346
                    {+0.2918238639831543d, +9.979706999541057E-9d,}, // 347
                    {+0.29255300760269165d, -4.916314210412424E-9d,}, // 348
                    {+0.293281614780426d, -1.4611908070155308E-8d,}, // 349
                    {+0.2940096855163574d, -1.833351586679361E-8d,}, // 350
                    {+0.29473721981048584d, -1.530926726615185E-8d,}, // 351
                    {+0.2954642176628113d, -4.7689754029101934E-9d,}, // 352
                    {+0.29619067907333374d, +1.4055868011423819E-8d,}, // 353
                    {+0.296916663646698d, -1.7672547212604003E-8d,}, // 354
                    {+0.2976420521736145d, +2.0020234215759705E-8d,}, // 355
                    {+0.2983669638633728d, +8.688424478730524E-9d,}, // 356
                    {+0.2990913391113281d, +8.69851089918337E-9d,}, // 357
                    {+0.29981517791748047d, +2.0810681643102672E-8d,}, // 358
                    {+0.3005385398864746d, -1.3821169493779352E-8d,}, // 359
                    {+0.301261305809021d, +2.4769140784919128E-8d,}, // 360
                    {+0.3019835948944092d, +1.8127576600610336E-8d,}, // 361
                    {+0.3027053475379944d, +2.6612401062437074E-8d,}, // 362
                    {+0.3034266233444214d, -8.629042891789934E-9d,}, // 363
                    {+0.3041473627090454d, -2.724174869314043E-8d,}, // 364
                    {+0.30486756563186646d, -2.8476975783775358E-8d,}, // 365
                    {+0.3055872321128845d, -1.1587600174449919E-8d,}, // 366
                    {+0.3063063621520996d, +2.417189020581056E-8d,}, // 367
                    {+0.3070250153541565d, +1.99407553679345E-8d,}, // 368
                    {+0.3077431917190552d, -2.35387025694381E-8d,}, // 369
                    {+0.3084607720375061d, +1.3683509995845583E-8d,}, // 370
                    {+0.30917787551879883d, +1.3137214081023085E-8d,}, // 371
                    {+0.30989450216293335d, -2.444006866174775E-8d,}, // 372
                    {+0.3106105327606201d, +2.0896888605749563E-8d,}, // 373
                    {+0.31132614612579346d, -2.893149098508887E-8d,}, // 374
                    {+0.31204116344451904d, +5.621509038251498E-9d,}, // 375
                    {+0.3127557039260864d, +6.0778104626050015E-9d,}, // 376
                    {+0.3134697675704956d, -2.6832941696716294E-8d,}, // 377
                    {+0.31418323516845703d, +2.6826625274495256E-8d,}, // 378
                    {+0.31489628553390503d, -1.1030897183911054E-8d,}, // 379
                    {+0.31560879945755005d, -2.047124671392676E-8d,}, // 380
                    {+0.3163207769393921d, -7.709990443086711E-10d,}, // 381
                    {+0.3170322775840759d, -1.0812918808112342E-8d,}, // 382
                    {+0.3177432417869568d, +9.727979174888975E-9d,}, // 383
                    {+0.31845372915267944d, +1.9658551724508715E-9d,}, // 384
                    {+0.3191636800765991d, +2.6222628001695826E-8d,}, // 385
                    {+0.3198731541633606d, +2.3609400272358744E-8d,}, // 386
                    {+0.32058215141296387d, -5.159602957634814E-9d,}, // 387
                    {+0.32129061222076416d, +2.329701319016099E-10d,}, // 388
                    {+0.32199859619140625d, -1.910633190395738E-8d,}, // 389
                    {+0.32270604372024536d, -2.863180390093667E-9d,}, // 390
                    {+0.32341301441192627d, -9.934041364456825E-9d,}, // 391
                    {+0.3241194486618042d, +1.999240777687192E-8d,}, // 392
                    {+0.3248254060745239d, +2.801670341647724E-8d,}, // 393
                    {+0.32553088665008545d, +1.4842534265191358E-8d,}, // 394
                    {+0.32623589038848877d, -1.882789920477354E-8d,}, // 395
                    {+0.3269403576850891d, -1.268923579073577E-8d,}, // 396
                    {+0.32764434814453125d, -2.564688370677835E-8d,}, // 397
                    {+0.3283478021621704d, +2.6015626820520968E-9d,}, // 398
                    {+0.32905077934265137d, +1.3147747907784344E-8d,}, // 399
                    {+0.3297532796859741d, +6.686493860720675E-9d,}, // 400
                    {+0.33045530319213867d, -1.608884086544153E-8d,}, // 401
                    {+0.33115679025650024d, +5.118287907840204E-9d,}, // 402
                    {+0.3318578004837036d, +1.139367970944884E-8d,}, // 403
                    {+0.3325583338737488d, +3.426327822115399E-9d,}, // 404
                    {+0.33325839042663574d, -1.809622142990733E-8d,}, // 405
                    {+0.3339579105377197d, +7.116780143398601E-9d,}, // 406
                    {+0.3346569538116455d, +2.0145352306345386E-8d,}, // 407
                    {+0.3353555202484131d, +2.167272474431968E-8d,}, // 408
                    {+0.33605360984802246d, +1.2380696294966822E-8d,}, // 409
                    {+0.33675122261047363d, -7.050361059209181E-9d,}, // 410
                    {+0.3374482989311218d, +2.366314656322868E-8d,}, // 411
                    {+0.3381449580192566d, -1.4010540194086646E-8d,}, // 412
                    {+0.3388410806655884d, -1.860165465666482E-10d,}, // 413
                    {+0.33953672647476196d, +6.206776940880773E-9d,}, // 414
                    {+0.34023189544677734d, +5.841137379010982E-9d,}, // 415
                    {+0.3409265875816345d, -6.11041311179286E-10d,}, // 416
                    {+0.3416208028793335d, -1.2479264502054702E-8d,}, // 417
                    {+0.34231454133987427d, -2.909443297645926E-8d,}, // 418
                    {+0.34300774335861206d, +9.815805717097634E-9d,}, // 419
                    {+0.3437005281448364d, -1.4291517981101049E-8d,}, // 420
                    {+0.3443927764892578d, +1.8457821628427503E-8d,}, // 421
                    {+0.34508460760116577d, -1.0481908869377813E-8d,}, // 422
                    {+0.34577590227127075d, +1.876076001514746E-8d,}, // 423
                    {+0.3464667797088623d, -1.2362653723769037E-8d,}, // 424
                    {+0.3471571207046509d, +1.6016578405624026E-8d,}, // 425
                    {+0.347847044467926d, -1.4652759033760925E-8d,}, // 426
                    {+0.3485364317893982d, +1.549533655901835E-8d,}, // 427
                    {+0.34922540187835693d, -1.2093068629412478E-8d,}, // 428
                    {+0.3499138355255127d, +2.244531711424792E-8d,}, // 429
                    {+0.35060185194015503d, +5.538565518604807E-10d,}, // 430
                    {+0.35128939151763916d, -1.7511499366215853E-8d,}, // 431
                    {+0.3519763946533203d, +2.850385787215544E-8d,}, // 432
                    {+0.35266298055648804d, +2.003926370146842E-8d,}, // 433
                    {+0.35334908962249756d, +1.734665280502264E-8d,}, // 434
                    {+0.3540347218513489d, +2.1071983674869414E-8d,}, // 435
                    {+0.35471993684768677d, -2.774475773922311E-8d,}, // 436
                    {+0.3554046154022217d, -9.250975291734664E-9d,}, // 437
                    {+0.3560888171195984d, +1.7590672330295415E-8d,}, // 438
                    {+0.35677260160446167d, -6.1837904549178745E-9d,}, // 439
                    {+0.35745590925216675d, -2.0330362973820856E-8d,}, // 440
                    {+0.3581387400627136d, -2.42109990366786E-8d,}, // 441
                    {+0.3588210940361023d, -1.7188958587407816E-8d,}, // 442
                    {+0.35950297117233276d, +1.3711958590112228E-9d,}, // 443
                    {+0.3601844310760498d, -2.7501042008405925E-8d,}, // 444
                    {+0.36086535453796387d, +1.6036460343275798E-8d,}, // 445
                    {+0.3615458607673645d, +1.3405964389498495E-8d,}, // 446
                    {+0.36222589015960693d, +2.484237749027735E-8d,}, // 447
                    {+0.36290550231933594d, -8.629967484362177E-9d,}, // 448
                    {+0.36358463764190674d, -2.6778729562324134E-8d,}, // 449
                    {+0.36426329612731934d, -2.8977490516960565E-8d,}, // 450
                    {+0.36494147777557373d, -1.4601106624823502E-8d,}, // 451
                    {+0.3656191825866699d, +1.69742947894444E-8d,}, // 452
                    {+0.3662964701652527d, +6.7666740211281175E-9d,}, // 453
                    {+0.36697328090667725d, +1.500201674336832E-8d,}, // 454
                    {+0.3676496744155884d, -1.730424167425052E-8d,}, // 455
                    {+0.36832553148269653d, +2.9676011119845104E-8d,}, // 456
                    {+0.36900103092193604d, -2.2253590346826743E-8d,}, // 457
                    {+0.36967599391937256d, +6.3372065441089185E-9d,}, // 458
                    {+0.37035053968429565d, -3.145816653215968E-9d,}, // 459
                    {+0.37102460861206055d, +9.515812117036965E-9d,}, // 460
                    {+0.371698260307312d, -1.4669965113042639E-8d,}, // 461
                    {+0.3723714351654053d, -1.548715389333397E-8d,}, // 462
                    {+0.37304413318634033d, +7.674361647125109E-9d,}, // 463
                    {+0.37371641397476196d, -4.181177882069608E-9d,}, // 464
                    {+0.3743882179260254d, +9.158530500130718E-9d,}, // 465
                    {+0.3750596046447754d, -1.13047236597869E-8d,}, // 466
                    {+0.3757305145263672d, -5.36108186384227E-9d,}, // 467
                    {+0.3764009475708008d, +2.7593452284747873E-8d,}, // 468
                    {+0.37707096338272095d, +2.8557016344085205E-8d,}, // 469
                    {+0.3777405619621277d, -1.868818164036E-9d,}, // 470
                    {+0.3784096837043762d, -3.479042513414447E-9d,}, // 471
                    {+0.37907832860946655d, +2.432550290565648E-8d,}, // 472
                    {+0.37974655628204346d, +2.2538131805476768E-8d,}, // 473
                    {+0.38041436672210693d, -8.244395239939089E-9d,}, // 474
                    {+0.3810817003250122d, -7.821867597227376E-9d,}, // 475
                    {+0.3817485570907593d, +2.4400089062515914E-8d,}, // 476
                    {+0.3824149966239929d, +2.9410015940087773E-8d,}, // 477
                    {+0.38308101892471313d, +7.799913824734797E-9d,}, // 478
                    {+0.38374656438827515d, +1.976524624939355E-8d,}, // 479
                    {+0.38441169261932373d, +6.291008309266035E-9d,}, // 480
                    {+0.3850763440132141d, +2.757030889767851E-8d,}, // 481
                    {+0.38574057817459106d, +2.4585794728405612E-8d,}, // 482
                    {+0.3864043951034546d, -2.0764122246389383E-9d,}, // 483
                    {+0.3870677351951599d, +7.77328837578952E-9d,}, // 484
                    {+0.3877306580543518d, -4.8859560029989374E-9d,}, // 485
                    {+0.3883931040763855d, +2.0133131420595028E-8d,}, // 486
                    {+0.38905513286590576d, +2.380738071335498E-8d,}, // 487
                    {+0.3897167444229126d, +6.7171126157142075E-9d,}, // 488
                    {+0.39037787914276123d, +2.9046141593926277E-8d,}, // 489
                    {+0.3910386562347412d, -2.7836800219410262E-8d,}, // 490
                    {+0.3916988968849182d, +1.545909820981726E-8d,}, // 491
                    {+0.39235877990722656d, -1.930436269002062E-8d,}, // 492
                    {+0.3930181860923767d, -1.2343297554921835E-8d,}, // 493
                    {+0.3936771750450134d, -2.268889128622553E-8d,}, // 494
                    {+0.39433568716049194d, +9.835827818608177E-9d,}, // 495
                    {+0.39499378204345703d, +2.6197411946856397E-8d,}, // 496
                    {+0.3956514596939087d, +2.6965931069318893E-8d,}, // 497
                    {+0.3963087201118469d, +1.2710331127772166E-8d,}, // 498
                    {+0.39696556329727173d, -1.6001563011916016E-8d,}, // 499
                    {+0.39762192964553833d, +1.0016001590267064E-9d,}, // 500
                    {+0.3982778787612915d, +4.680767399874334E-9d,}, // 501
                    {+0.39893341064453125d, -4.399582029272418E-9d,}, // 502
                    {+0.39958852529525757d, -2.5676078228301587E-8d,}, // 503
                    {+0.4002431631088257d, +1.0181870233355787E-9d,}, // 504
                    {+0.40089738368988037d, +1.6639728835984655E-8d,}, // 505
                    {+0.40155118703842163d, +2.174860642202632E-8d,}, // 506
                    {+0.40220457315444946d, +1.6903781197123503E-8d,}, // 507
                    {+0.40285754203796387d, +2.663119647467697E-9d,}, // 508
                    {+0.40351009368896484d, -2.0416603812329616E-8d,}, // 509
                    {+0.4041621685028076d, +7.82494078472695E-9d,}, // 510
                    {+0.40481382608413696d, +2.833770747113627E-8d,}, // 511
                    {+0.40546512603759766d, -1.7929433274271985E-8d,}, // 512
                    {+0.40611594915390015d, -1.1214757379328965E-8d,}, // 513
                    {+0.4067663550376892d, -1.0571553019207106E-8d,}, // 514
                    {+0.40741634368896484d, -1.5449538712332313E-8d,}, // 515
                    {+0.40806591510772705d, -2.529950530235105E-8d,}, // 516
                    {+0.40871500968933105d, +2.0031331601617008E-8d,}, // 517
                    {+0.4093637466430664d, +1.880755298741952E-9d,}, // 518
                    {+0.41001206636428833d, -1.9600580584843318E-8d,}, // 519
                    {+0.41065990924835205d, +1.573691633515306E-8d,}, // 520
                    {+0.4113073945045471d, -1.0772154376548336E-8d,}, // 521
                    {+0.411954402923584d, +2.0624330192486066E-8d,}, // 522
                    {+0.4126010537147522d, -8.741139170029572E-9d,}, // 523
                    {+0.4132472276687622d, +2.0881457123894216E-8d,}, // 524
                    {+0.41389304399490356d, -9.177488027521808E-9d,}, // 525
                    {+0.4145383834838867d, +2.0829952491625585E-8d,}, // 526
                    {+0.4151833653450012d, -7.767915492597301E-9d,}, // 527
                    {+0.4158278703689575d, +2.4774753446082082E-8d,}, // 528
                    {+0.41647201776504517d, -2.1581119071750435E-10d,}, // 529
                    {+0.4171157479286194d, -2.260047972865202E-8d,}, // 530
                    {+0.4177590012550354d, +1.775884601423381E-8d,}, // 531
                    {+0.41840189695358276d, +2.185301053838889E-9d,}, // 532
                    {+0.4190443754196167d, -9.185071463667081E-9d,}, // 533
                    {+0.4196864366531372d, -1.5821896727910552E-8d,}, // 534
                    {+0.4203280806541443d, -1.719582086188318E-8d,}, // 535
                    {+0.42096930742263794d, -1.2778508303324259E-8d,}, // 536
                    {+0.42161011695861816d, -2.042639194493364E-9d,}, // 537
                    {+0.42225050926208496d, +1.5538093219698803E-8d,}, // 538
                    {+0.4228905439376831d, -1.9115659590156936E-8d,}, // 539
                    {+0.42353010177612305d, +1.3729680248843432E-8d,}, // 540
                    {+0.42416930198669434d, -4.611893838830296E-9d,}, // 541
                    {+0.4248080849647522d, -1.4013456880651706E-8d,}, // 542
                    {+0.42544645071029663d, -1.3953728897042917E-8d,}, // 543
                    {+0.42608439922332764d, -3.912427573594197E-9d,}, // 544
                    {+0.4267219305038452d, +1.6629734283189315E-8d,}, // 545
                    {+0.42735910415649414d, -1.1413593493354881E-8d,}, // 546
                    {+0.42799586057662964d, -2.792046157580119E-8d,}, // 547
                    {+0.42863214015960693d, +2.723009182661306E-8d,}, // 548
                    {+0.42926812171936035d, -2.4260535621557444E-8d,}, // 549
                    {+0.42990362644195557d, -3.064060124024764E-9d,}, // 550
                    {+0.43053877353668213d, -2.787640178598121E-8d,}, // 551
                    {+0.4311734437942505d, +2.102412085257792E-8d,}, // 552
                    {+0.4318077564239502d, +2.4939635093999683E-8d,}, // 553
                    {+0.43244171142578125d, -1.5619414792273914E-8d,}, // 554
                    {+0.4330751895904541d, +1.9065734894871523E-8d,}, // 555
                    {+0.4337083101272583d, +1.0294301092654604E-8d,}, // 556
                    {+0.4343410134315491d, +1.8178469851136E-8d,}, // 557
                    {+0.4349733591079712d, -1.6379825102473853E-8d,}, // 558
                    {+0.4356052279472351d, +2.6334323946685834E-8d,}, // 559
                    {+0.43623673915863037d, +2.761628769925529E-8d,}, // 560
                    {+0.436867892742157d, -1.2030229087793677E-8d,}, // 561
                    {+0.4374985694885254d, +2.7106814809424793E-8d,}, // 562
                    {+0.43812888860702515d, +2.631993083235205E-8d,}, // 563
                    {+0.43875885009765625d, -1.3890028312254422E-8d,}, // 564
                    {+0.43938833475112915d, +2.6186133735555794E-8d,}, // 565
                    {+0.4400174617767334d, +2.783809071694788E-8d,}, // 566
                    {+0.440646231174469d, -8.436135220472006E-9d,}, // 567
                    {+0.44127458333969116d, -2.2534815932619883E-8d,}, // 568
                    {+0.4419025182723999d, -1.3961804471714283E-8d,}, // 569
                    {+0.4425300359725952d, +1.7778112039716255E-8d,}, // 570
                    {+0.4431571960449219d, +1.3574569976673652E-8d,}, // 571
                    {+0.4437839984893799d, -2.607907890164073E-8d,}, // 572
                    {+0.4444103240966797d, +1.8518879652136628E-8d,}, // 573
                    {+0.44503629207611084d, +2.865065604247164E-8d,}, // 574
                    {+0.44566190242767334d, +4.806827797299427E-9d,}, // 575
                    {+0.4462870955467224d, +7.0816970994232115E-9d,}, // 576
                    {+0.44691193103790283d, -2.3640641240074437E-8d,}, // 577
                    {+0.4475363492965698d, -2.7267718387865538E-8d,}, // 578
                    {+0.4481603503227234d, -3.3126235292976077E-9d,}, // 579
                    {+0.4487839937210083d, -1.0894001590268427E-8d,}, // 580
                    {+0.4494072198867798d, +1.0077883359971829E-8d,}, // 581
                    {+0.4500300884246826d, +4.825712712114668E-10d,}, // 582
                    {+0.450652539730072d, +2.0407987470746858E-8d,}, // 583
                    {+0.4512746334075928d, +1.073186581170719E-8d,}, // 584
                    {+0.4518963694572449d, -2.8064314757880205E-8d,}, // 585
                    {+0.45251762866973877d, +2.3709316816226527E-8d,}, // 586
                    {+0.4531385898590088d, -1.2281487504266522E-8d,}, // 587
                    {+0.4537591338157654d, -1.634864487421458E-8d,}, // 588
                    {+0.45437926054000854d, +1.1985747222409522E-8d,}, // 589
                    {+0.45499902963638306d, +1.3594057956219485E-8d,}, // 590
                    {+0.4556184411048889d, -1.1047585095328619E-8d,}, // 591
                    {+0.45623743534088135d, -1.8592937532754405E-9d,}, // 592
                    {+0.4568560719490051d, -1.797135137545755E-8d,}, // 593
                    {+0.4574742913246155d, +6.943684261645378E-10d,}, // 594
                    {+0.4580921530723572d, -4.994175141684681E-9d,}, // 595
                    {+0.45870959758758545d, +2.5039391215625133E-8d,}, // 596
                    {+0.45932674407958984d, -2.7943366835352838E-8d,}, // 597
                    {+0.45994341373443604d, +1.534146910128904E-8d,}, // 598
                    {+0.46055978536605835d, -2.3450920230816267E-8d,}, // 599
                    {+0.46117573976516724d, -2.4642997069960124E-8d,}, // 600
                    {+0.4617912769317627d, +1.2232622070370946E-8d,}, // 601
                    {+0.4624064564704895d, +2.80378133047839E-8d,}, // 602
                    {+0.46302127838134766d, +2.3238237048117092E-8d,}, // 603
                    {+0.46363574266433716d, -1.7013046451109475E-9d,}, // 604
                    {+0.46424978971481323d, +1.3287778803035383E-8d,}, // 605
                    {+0.46486347913742065d, +9.06393426961373E-9d,}, // 606
                    {+0.4654768109321594d, -1.3910598647592876E-8d,}, // 607
                    {+0.46608972549438477d, +4.430214458933614E-9d,}, // 608
                    {+0.46670228242874146d, +4.942270562885745E-9d,}, // 609
                    {+0.4673144817352295d, -1.1914734393460718E-8d,}, // 610
                    {+0.4679262638092041d, +1.3922696570638494E-8d,}, // 611
                    {+0.46853768825531006d, +2.3307929211781914E-8d,}, // 612
                    {+0.46914875507354736d, +1.669813444584674E-8d,}, // 613
                    {+0.469759464263916d, -5.450354376430758E-9d,}, // 614
                    {+0.47036975622177124d, +1.6922605350647674E-8d,}, // 615
                    {+0.4709796905517578d, +2.4667033200046904E-8d,}, // 616
                    {+0.47158926725387573d, +1.8236762070433784E-8d,}, // 617
                    {+0.472198486328125d, -1.915204563140137E-9d,}, // 618
                    {+0.47280728816986084d, +2.426795414605756E-8d,}, // 619
                    {+0.4734157919883728d, -2.19717006713618E-8d,}, // 620
                    {+0.47402387857437134d, -2.0974352165535873E-8d,}, // 621
                    {+0.47463154792785645d, +2.770970558184228E-8d,}, // 622
                    {+0.4752389192581177d, +5.32006955298355E-9d,}, // 623
                    {+0.47584593296051025d, -2.809054633964104E-8d,}, // 624
                    {+0.4764525294303894d, -1.2470243596102937E-8d,}, // 625
                    {+0.4770587682723999d, -6.977226702440138E-9d,}, // 626
                    {+0.47766464948654175d, -1.1165866833118273E-8d,}, // 627
                    {+0.47827017307281494d, -2.4591344661022708E-8d,}, // 628
                    {+0.4788752794265747d, +1.2794996377383974E-8d,}, // 629
                    {+0.4794800877571106d, -1.7772927065973874E-8d,}, // 630
                    {+0.48008447885513306d, +3.35657712457243E-9d,}, // 631
                    {+0.48068851232528687d, +1.7020465042442242E-8d,}, // 632
                    {+0.481292188167572d, +2.365953779624783E-8d,}, // 633
                    {+0.4818955063819885d, +2.3713798664443718E-8d,}, // 634
                    {+0.4824984669685364d, +1.7622455019548098E-8d,}, // 635
                    {+0.4831010699272156d, +5.823920246566496E-9d,}, // 636
                    {+0.4837033152580261d, -1.1244184344361017E-8d,}, // 637
                    {+0.48430514335632324d, +2.645961716432205E-8d,}, // 638
                    {+0.4849066734313965d, +1.6207809718247905E-10d,}, // 639
                    {+0.4855077862739563d, +2.9507744508973654E-8d,}, // 640
                    {+0.48610860109329224d, -4.278201128741098E-9d,}, // 641
                    {+0.48670899868011475d, +1.844722015961139E-8d,}, // 642
                    {+0.4873090982437134d, -2.1092372471088425E-8d,}, // 643
                    {+0.4879087805747986d, -3.2555596107382053E-9d,}, // 644
                    {+0.48850810527801514d, +1.2784366845429667E-8d,}, // 645
                    {+0.48910707235336304d, +2.7457984659996047E-8d,}, // 646
                    {+0.48970574140548706d, -1.8409546441412518E-8d,}, // 647
                    {+0.49030399322509766d, -5.179903818099661E-9d,}, // 648
                    {+0.4909018874168396d, +7.97053127828682E-9d,}, // 649
                    {+0.4914994239807129d, +2.146925464473481E-8d,}, // 650
                    {+0.4920966625213623d, -2.3861648589988232E-8d,}, // 651
                    {+0.4926934838294983d, -8.386923035320549E-9d,}, // 652
                    {+0.4932899475097656d, +8.713990131749256E-9d,}, // 653
                    {+0.4938860535621643d, +2.7865534085810115E-8d,}, // 654
                    {+0.4944818615913391d, -1.011325138560159E-8d,}, // 655
                    {+0.4950772523880005d, +1.4409851026316708E-8d,}, // 656
                    {+0.495672345161438d, -1.735227547472004E-8d,}, // 657
                    {+0.49626702070236206d, +1.4231078209064581E-8d,}, // 658
                    {+0.49686139822006226d, -9.628709342929729E-9d,}, // 659
                    {+0.4974554181098938d, -2.8907074856577267E-8d,}, // 660
                    {+0.4980490207672119d, +1.6419797090870802E-8d,}, // 661
                    {+0.49864232540130615d, +7.561041519403049E-9d,}, // 662
                    {+0.49923527240753174d, +4.538983468118194E-9d,}, // 663
                    {+0.49982786178588867d, +7.770560657946324E-9d,}, // 664
                    {+0.500420093536377d, +1.767197002609876E-8d,}, // 665
                    {+0.5010119676589966d, +3.46586694799214E-8d,}, // 666
                    {+0.5016034841537476d, +5.914537964556077E-8d,}, // 667
                    {+0.5021947622299194d, -2.7663203939320167E-8d,}, // 668
                    {+0.5027855634689331d, +1.3064749115929298E-8d,}, // 669
                    {+0.5033761262893677d, -5.667682106730711E-8d,}, // 670
                    {+0.503966212272644d, +1.9424534974370594E-9d,}, // 671
                    {+0.5045560598373413d, -4.908494602153544E-8d,}, // 672
                    {+0.5051454305648804d, +2.906989285008994E-8d,}, // 673
                    {+0.5057345628738403d, -1.602000800745108E-9d,}, // 674
                    {+0.5063233375549316d, -2.148245271118002E-8d,}, // 675
                    {+0.5069117546081543d, -3.016329994276181E-8d,}, // 676
                    {+0.5074998140335083d, -2.7237099632871992E-8d,}, // 677
                    {+0.5080875158309937d, -1.2297127301923986E-8d,}, // 678
                    {+0.5086748600006104d, +1.5062624834468093E-8d,}, // 679
                    {+0.5092618465423584d, +5.524744954836658E-8d,}, // 680
                    {+0.5098485946655273d, -1.054736327333046E-8d,}, // 681
                    {+0.5104348659515381d, +5.650063324725722E-8d,}, // 682
                    {+0.5110208988189697d, +1.8376017791642605E-8d,}, // 683
                    {+0.5116065740585327d, -5.309470636324855E-9d,}, // 684
                    {+0.512191891670227d, -1.4154089255217218E-8d,}, // 685
                    {+0.5127768516540527d, -7.756800301729815E-9d,}, // 686
                    {+0.5133614540100098d, +1.4282730618002001E-8d,}, // 687
                    {+0.5139456987380981d, +5.2364136172269755E-8d,}, // 688
                    {+0.5145297050476074d, -1.2322940607922115E-8d,}, // 689
                    {+0.5151132345199585d, +5.903831350855322E-8d,}, // 690
                    {+0.5156965255737305d, +2.8426856726994483E-8d,}, // 691
                    {+0.5162794589996338d, +1.544882070711032E-8d,}, // 692
                    {+0.5168620347976685d, +2.0500353979930155E-8d,}, // 693
                    {+0.5174442529678345d, +4.397691311390564E-8d,}, // 694
                    {+0.5180262327194214d, -3.2936025225250634E-8d,}, // 695
                    {+0.5186077356338501d, +2.857419553449673E-8d,}, // 696
                    {+0.5191890001296997d, -9.51761338269325E-9d,}, // 697
                    {+0.5197699069976807d, -2.7609457648450225E-8d,}, // 698
                    {+0.520350456237793d, -2.5309316441333305E-8d,}, // 699
                    {+0.5209306478500366d, -2.2258513086839407E-9d,}, // 700
                    {+0.5215104818344116d, +4.203159541613745E-8d,}, // 701
                    {+0.5220900774002075d, -1.1356287358852729E-8d,}, // 702
                    {+0.5226693153381348d, -4.279090925831093E-8d,}, // 703
                    {+0.5232481956481934d, -5.188364552285819E-8d,}, // 704
                    {+0.5238267183303833d, -3.82465458937857E-8d,}, // 705
                    {+0.5244048833847046d, -1.4923330530645769E-9d,}, // 706
                    {+0.5249826908111572d, +5.8765598932137004E-8d,}, // 707
                    {+0.5255602598190308d, +2.3703896609663678E-8d,}, // 708
                    {+0.5261374711990356d, +1.2917117341231647E-8d,}, // 709
                    {+0.5267143249511719d, +2.6789862192139226E-8d,}, // 710
                    {+0.527290940284729d, -5.350322253112414E-8d,}, // 711
                    {+0.5278670787811279d, +1.0839714455426386E-8d,}, // 712
                    {+0.5284429788589478d, -1.821729591343314E-8d,}, // 713
                    {+0.5290185213088989d, -2.1083014672301448E-8d,}, // 714
                    {+0.5295937061309814d, +2.623848491704216E-9d,}, // 715
                    {+0.5301685333251953d, +5.328392630534142E-8d,}, // 716
                    {+0.5307431221008301d, +1.206790586971942E-8d,}, // 717
                    {+0.5313173532485962d, -1.4356011804377797E-9d,}, // 718
                    {+0.5318912267684937d, +1.3152074173459994E-8d,}, // 719
                    {+0.5324647426605225d, +5.6208949382936426E-8d,}, // 720
                    {+0.5330380201339722d, +8.90310227565917E-9d,}, // 721
                    {+0.5336109399795532d, -9.179458802504127E-9d,}, // 722
                    {+0.5341835021972656d, +2.337337845617735E-9d,}, // 723
                    {+0.5347557067871094d, +4.3828918300477925E-8d,}, // 724
                    {+0.535327672958374d, -3.5392250480081715E-9d,}, // 725
                    {+0.53589928150177d, -2.0183663375378704E-8d,}, // 726
                    {+0.5364705324172974d, -5.730898606435436E-9d,}, // 727
                    {+0.537041425704956d, +4.0191927599879235E-8d,}, // 728
                    {+0.5376120805740356d, -1.2522542401353875E-9d,}, // 729
                    {+0.5381823778152466d, -1.0482571326594316E-8d,}, // 730
                    {+0.5387523174285889d, +1.2871924223480165E-8d,}, // 731
                    {+0.539322018623352d, -5.002774317612589E-8d,}, // 732
                    {+0.539891242980957d, +3.960668706590162E-8d,}, // 733
                    {+0.5404602289199829d, +4.372568630242375E-8d,}, // 734
                    {+0.5410289764404297d, -3.730232461206926E-8d,}, // 735
                    {+0.5415972471237183d, +3.5309026109857795E-8d,}, // 736
                    {+0.5421652793884277d, +2.3508325311148225E-8d,}, // 737
                    {+0.5427329540252686d, +4.6871403168921666E-8d,}, // 738
                    {+0.5433003902435303d, -1.3445113140270216E-8d,}, // 739
                    {+0.5438674688339233d, -3.786663982218041E-8d,}, // 740
                    {+0.5444341897964478d, -2.602850370608209E-8d,}, // 741
                    {+0.5450005531311035d, +2.2433348713144506E-8d,}, // 742
                    {+0.5455666780471802d, -1.1326936872620137E-8d,}, // 743
                    {+0.5461324453353882d, -7.737252533211342E-9d,}, // 744
                    {+0.5466978549957275d, +3.3564604642699844E-8d,}, // 745
                    {+0.5472630262374878d, -6.269066061111782E-9d,}, // 746
                    {+0.5478278398513794d, -7.667998948729528E-9d,}, // 747
                    {+0.5483922958374023d, +2.9728170818998143E-8d,}, // 748
                    {+0.5489565134048462d, -1.2930091396008281E-8d,}, // 749
                    {+0.5495203733444214d, -1.607434968107079E-8d,}, // 750
                    {+0.5500838756561279d, +2.0653935146671156E-8d,}, // 751
                    {+0.5506471395492554d, -2.1596593091833788E-8d,}, // 752
                    {+0.5512100458145142d, -2.3259315921149476E-8d,}, // 753
                    {+0.5517725944519043d, +1.6022492496522704E-8d,}, // 754
                    {+0.5523349046707153d, -2.260433328226171E-8d,}, // 755
                    {+0.5528968572616577d, -1.957497997726303E-8d,}, // 756
                    {+0.5534584522247314d, +2.5465477111883854E-8d,}, // 757
                    {+0.5540198087692261d, -6.33792454933092E-9d,}, // 758
                    {+0.554580807685852d, +4.577835263278281E-9d,}, // 759
                    {+0.5551414489746094d, +5.856589221771548E-8d,}, // 760
                    {+0.5557018518447876d, +3.6769498759522324E-8d,}, // 761
                    {+0.5562618970870972d, +5.874989409410614E-8d,}, // 762
                    {+0.5568217039108276d, +5.649147309876989E-9d,}, // 763
                    {+0.5573811531066895d, -2.9726830960751796E-9d,}, // 764
                    {+0.5579402446746826d, +3.323458344853057E-8d,}, // 765
                    {+0.5584990978240967d, -4.588749093664028E-9d,}, // 766
                    {+0.5590575933456421d, +3.115616594184543E-9d,}, // 767
                    {+0.5596157312393188d, +5.6696103838614634E-8d,}, // 768
                    {+0.5601736307144165d, +3.7291263280048303E-8d,}, // 769
                    {+0.5607312917709351d, -5.4751646725093355E-8d,}, // 770
                    {+0.5612884759902954d, +1.9332630743320287E-8d,}, // 771
                    {+0.5618454217910767d, +2.147161515775941E-8d,}, // 772
                    {+0.5624021291732788d, -4.7989172862560625E-8d,}, // 773
                    {+0.5629583597183228d, +4.971378973445109E-8d,}, // 774
                    {+0.5635144710540771d, -4.2702997139152675E-8d,}, // 775
                    {+0.5640701055526733d, +3.273212962622764E-8d,}, // 776
                    {+0.5646255016326904d, +3.79438125545842E-8d,}, // 777
                    {+0.5651806592941284d, -2.6725298288329835E-8d,}, // 778
                    {+0.5657354593276978d, -4.1723833577410244E-8d,}, // 779
                    {+0.5662899017333984d, -6.71028256490915E-9d,}, // 780
                    {+0.56684410572052d, -4.055299181908475E-8d,}, // 781
                    {+0.567397952079773d, -2.3702295314000405E-8d,}, // 782
                    {+0.5679514408111572d, +4.4181618172507453E-8d,}, // 783
                    {+0.5685046911239624d, +4.4228706309734985E-8d,}, // 784
                    {+0.5690577030181885d, -2.3222346436879016E-8d,}, // 785
                    {+0.5696103572845459d, -3.862412756175274E-8d,}, // 786
                    {+0.5701626539230347d, -1.6390743801589046E-9d,}, // 787
                    {+0.5707147121429443d, -3.1139472791083883E-8d,}, // 788
                    {+0.5712664127349854d, -7.579587391156013E-9d,}, // 789
                    {+0.5718178749084473d, -4.983281844744412E-8d,}, // 790
                    {+0.5723689794540405d, -3.835454246739619E-8d,}, // 791
                    {+0.5729197263717651d, +2.7190020372374008E-8d,}, // 792
                    {+0.5734702348709106d, +2.7925807446276126E-8d,}, // 793
                    {+0.574020504951477d, -3.5813506001861646E-8d,}, // 794
                    {+0.5745704174041748d, -4.448550564530588E-8d,}, // 795
                    {+0.5751199722290039d, +2.2423840341717488E-9d,}, // 796
                    {+0.5756692886352539d, -1.450709904687712E-8d,}, // 797
                    {+0.5762182474136353d, +2.4806815282282017E-8d,}, // 798
                    {+0.5767669677734375d, +1.3057724436551892E-9d,}, // 799
                    {+0.5773153305053711d, +3.4529452510568104E-8d,}, // 800
                    {+0.5778634548187256d, +5.598413198183808E-9d,}, // 801
                    {+0.5784112215042114d, +3.405124925700107E-8d,}, // 802
                    {+0.5789587497711182d, +1.0074354568442952E-9d,}, // 803
                    {+0.5795059204101562d, +2.600448597385527E-8d,}, // 804
                    {+0.5800528526306152d, -9.83920263200211E-9d,}, // 805
                    {+0.5805994272232056d, +1.3012807963586057E-8d,}, // 806
                    {+0.5811457633972168d, -2.432215917965441E-8d,}, // 807
                    {+0.5816917419433594d, -2.308736892479391E-9d,}, // 808
                    {+0.5822374820709229d, -3.983067093146514E-8d,}, // 809
                    {+0.5827828645706177d, -1.735366061128156E-8d,}, // 810
                    {+0.5833280086517334d, -5.376251584638963E-8d,}, // 811
                    {+0.5838727951049805d, -2.952399778965259E-8d,}, // 812
                    {+0.5844172239303589d, +5.5685313670430624E-8d,}, // 813
                    {+0.5849615335464478d, -3.6230268489088716E-8d,}, // 814
                    {+0.5855053663253784d, +5.267948957869391E-8d,}, // 815
                    {+0.5860490798950195d, -3.489144132234588E-8d,}, // 816
                    {+0.5865923166275024d, +5.9006122320612716E-8d,}, // 817
                    {+0.5871354341506958d, -2.2934896740542648E-8d,}, // 818
                    {+0.5876781940460205d, -4.1975650319859075E-8d,}, // 819
                    {+0.5882205963134766d, +2.2036094805348692E-9d,}, // 820
                    {+0.5887627601623535d, -9.287179048539306E-9d,}, // 821
                    {+0.5893045663833618d, +4.3079982556221595E-8d,}, // 822
                    {+0.589846134185791d, +4.041399585161321E-8d,}, // 823
                    {+0.5903874635696411d, -1.696746473863933E-8d,}, // 824
                    {+0.5909284353256226d, -9.53795080582038E-9d,}, // 825
                    {+0.5914691686630249d, -5.619010749352923E-8d,}, // 826
                    {+0.5920095443725586d, -3.7398514182529506E-8d,}, // 827
                    {+0.5925495624542236d, +4.71524479659295E-8d,}, // 828
                    {+0.5930894613265991d, -4.0640692434639215E-8d,}, // 829
                    {+0.5936288833618164d, +5.716453096255401E-8d,}, // 830
                    {+0.5941681861877441d, -1.6745661720946737E-8d,}, // 831
                    {+0.5947071313858032d, -2.3639110433141897E-8d,}, // 832
                    {+0.5952457189559937d, +3.67972590471072E-8d,}, // 833
                    {+0.595784068107605d, +4.566672575206695E-8d,}, // 834
                    {+0.5963221788406372d, +3.2813537149653483E-9d,}, // 835
                    {+0.5968599319458008d, +2.916199305533732E-8d,}, // 836
                    {+0.5973974466323853d, +4.410412409109416E-9d,}, // 837
                    {+0.5979346036911011d, +4.85464582112459E-8d,}, // 838
                    {+0.5984715223312378d, +4.267089756924666E-8d,}, // 839
                    {+0.5990082025527954d, -1.2906712010774655E-8d,}, // 840
                    {+0.5995445251464844d, +1.3319784467641742E-9d,}, // 841
                    {+0.6000806093215942d, -3.35137581974451E-8d,}, // 842
                    {+0.6006163358688354d, +2.0734340706476473E-9d,}, // 843
                    {+0.6011518239974976d, -1.0808162722402073E-8d,}, // 844
                    {+0.601686954498291d, +4.735781872502109E-8d,}, // 845
                    {+0.6022218465805054d, +5.76686738430634E-8d,}, // 846
                    {+0.6027565002441406d, +2.043049589651736E-8d,}, // 847
                    {+0.6032907962799072d, +5.515817703577808E-8d,}, // 848
                    {+0.6038248538970947d, +4.2947540692649586E-8d,}, // 849
                    {+0.6043586730957031d, -1.589678872195875E-8d,}, // 850
                    {+0.6048921346664429d, -1.8613847754677912E-9d,}, // 851
                    {+0.6054253578186035d, -3.3851886626187444E-8d,}, // 852
                    {+0.6059582233428955d, +7.64416021682279E-9d,}, // 853
                    {+0.6064908504486084d, +3.7201467248814224E-9d,}, // 854
                    {+0.6070232391357422d, -4.532172996647129E-8d,}, // 855
                    {+0.6075552701950073d, -1.997046552871766E-8d,}, // 856
                    {+0.6080870628356934d, -3.913411606668587E-8d,}, // 857
                    {+0.6086184978485107d, +1.6697361107868944E-8d,}, // 858
                    {+0.609149694442749d, +2.8614950293715483E-8d,}, // 859
                    {+0.6096806526184082d, -3.081552929643174E-9d,}, // 860
                    {+0.6102112531661987d, +4.111645931319645E-8d,}, // 861
                    {+0.6107416152954102d, +4.2298539553668435E-8d,}, // 862
                    {+0.6112717390060425d, +7.630546413718035E-10d,}, // 863
                    {+0.6118015050888062d, +3.601718675118614E-8d,}, // 864
                    {+0.6123310327529907d, +2.914906573537692E-8d,}, // 865
                    {+0.6128603219985962d, -1.9544361222269494E-8d,}, // 866
                    {+0.613389253616333d, +9.442671392695732E-9d,}, // 867
                    {+0.6139179468154907d, -2.8031202304593286E-9d,}, // 868
                    {+0.6144464015960693d, -5.598619958143586E-8d,}, // 869
                    {+0.6149744987487793d, -3.060220883766096E-8d,}, // 870
                    {+0.6155023574829102d, -4.556583652800433E-8d,}, // 871
                    {+0.6160298585891724d, +1.8626341656366314E-8d,}, // 872
                    {+0.6165571212768555d, +4.305870564227991E-8d,}, // 873
                    {+0.6170841455459595d, +2.8024460607734262E-8d,}, // 874
                    {+0.6176109313964844d, -2.6183651590639875E-8d,}, // 875
                    {+0.6181373596191406d, -6.406189112730307E-11d,}, // 876
                    {+0.6186635494232178d, -1.2534241706168776E-8d,}, // 877
                    {+0.6191893815994263d, +5.5906456251308664E-8d,}, // 878
                    {+0.6197150945663452d, -3.286964881802063E-8d,}, // 879
                    {+0.6202404499053955d, -4.0153537978961E-8d,}, // 880
                    {+0.6207654476165771d, +3.434477109643361E-8d,}, // 881
                    {+0.6212903261184692d, -4.750377491075032E-8d,}, // 882
                    {+0.6218148469924927d, -4.699152670372743E-8d,}, // 883
                    {+0.6223390102386475d, +3.617013128065961E-8d,}, // 884
                    {+0.6228630542755127d, -3.6149218175202596E-8d,}, // 885
                    {+0.6233867406845093d, -2.5243286814648133E-8d,}, // 886
                    {+0.6239101886749268d, -5.003410681432538E-8d,}, // 887
                    {+0.6244332790374756d, +8.974417915105033E-9d,}, // 888
                    {+0.6249561309814453d, +3.285935446876949E-8d,}, // 889
                    {+0.6254787445068359d, +2.190661054038537E-8d,}, // 890
                    {+0.6260011196136475d, -2.3598354190515998E-8d,}, // 891
                    {+0.6265231370925903d, +1.5838762427747586E-8d,}, // 892
                    {+0.6270449161529541d, +2.129323729978037E-8d,}, // 893
                    {+0.6275664567947388d, -6.950808333865794E-9d,}, // 894
                    {+0.6280876398086548d, +5.059959203156465E-8d,}, // 895
                    {+0.6286087036132812d, -4.41909071122557E-8d,}, // 896
                    {+0.6291294097900391d, -5.262093550784066E-8d,}, // 897
                    {+0.6296497583389282d, +2.559185648444699E-8d,}, // 898
                    {+0.6301699876785278d, -4.768920119497491E-8d,}, // 899
                    {+0.6306898593902588d, -3.376406008397877E-8d,}, // 900
                    {+0.6312094926834106d, -5.156097914033476E-8d,}, // 901
                    {+0.6317287683486938d, +1.840992392368355E-8d,}, // 902
                    {+0.632247805595398d, +5.721951534729663E-8d,}, // 903
                    {+0.6327667236328125d, -5.406177467045421E-8d,}, // 904
                    {+0.6332851648330688d, +4.247320713683124E-8d,}, // 905
                    {+0.6338034868240356d, -1.0524557502830645E-8d,}, // 906
                    {+0.6343214511871338d, +2.5641927558519502E-8d,}, // 907
                    {+0.6348391771316528d, +3.204135737993823E-8d,}, // 908
                    {+0.6353566646575928d, +8.951285029786536E-9d,}, // 909
                    {+0.6358739137649536d, -4.335116707228395E-8d,}, // 910
                    {+0.6363908052444458d, -5.380016714089483E-9d,}, // 911
                    {+0.6369074583053589d, +3.931710344901743E-9d,}, // 912
                    {+0.6374238729476929d, -1.5140150088220166E-8d,}, // 913
                    {+0.6379399299621582d, +5.688910024377372E-8d,}, // 914
                    {+0.638455867767334d, -1.8124135273572568E-8d,}, // 915
                    {+0.6389714479446411d, -1.486720391901626E-9d,}, // 916
                    {+0.6394867897033691d, -1.2133811978747018E-8d,}, // 917
                    {+0.6400018930435181d, -4.9791700939901716E-8d,}, // 918
                    {+0.6405166387557983d, +5.022188652837274E-9d,}, // 919
                    {+0.6410311460494995d, +3.337143177933685E-8d,}, // 920
                    {+0.6415454149246216d, +3.55284719912458E-8d,}, // 921
                    {+0.6420594453811646d, +1.1765332726757802E-8d,}, // 922
                    {+0.6425732374191284d, -3.7646381826067834E-8d,}, // 923
                    {+0.6430866718292236d, +6.773803682579552E-9d,}, // 924
                    {+0.6435998678207397d, +2.608736797081283E-8d,}, // 925
                    {+0.6441128253936768d, +2.056466263408266E-8d,}, // 926
                    {+0.6446255445480347d, -9.524376551107945E-9d,}, // 927
                    {+0.6451379060745239d, +5.5299060775883977E-8d,}, // 928
                    {+0.6456501483917236d, -2.3114497793159813E-8d,}, // 929
                    {+0.6461620330810547d, -6.077779731902102E-9d,}, // 930
                    {+0.6466736793518066d, -1.2531793589140273E-8d,}, // 931
                    {+0.6471850872039795d, -4.220866994206517E-8d,}, // 932
                    {+0.6476961374282837d, +2.4368339445199057E-8d,}, // 933
                    {+0.6482070684432983d, -5.095229574221907E-8d,}, // 934
                    {+0.6487176418304443d, -2.9485356677301627E-8d,}, // 935
                    {+0.6492279767990112d, -3.0173901411577916E-8d,}, // 936
                    {+0.649738073348999d, -5.275210583909726E-8d,}, // 937
                    {+0.6502478122711182d, +2.2254737134350224E-8d,}, // 938
                    {+0.6507574319839478d, -4.330693978322885E-8d,}, // 939
                    {+0.6512666940689087d, -1.0753950588009912E-8d,}, // 940
                    {+0.6517757177352905d, +9.686179886293545E-10d,}, // 941
                    {+0.6522845029830933d, -7.875434494414498E-9d,}, // 942
                    {+0.6527930498123169d, -3.702271091849158E-8d,}, // 943
                    {+0.6533012390136719d, +3.2999073763758614E-8d,}, // 944
                    {+0.6538093090057373d, -3.5966064858620067E-8d,}, // 945
                    {+0.6543170213699341d, -5.23735298540578E-9d,}, // 946
                    {+0.6548244953155518d, +6.237715351293023E-9d,}, // 947
                    {+0.6553317308425903d, -1.279462699936282E-9d,}, // 948
                    {+0.6558387279510498d, -2.7527887552743672E-8d,}, // 949
                    {+0.6563453674316406d, +4.696233317356646E-8d,}, // 950
                    {+0.6568518877029419d, -1.5967172745329108E-8d,}, // 951
                    {+0.6573580503463745d, +2.2361985518423144E-8d,}, // 952
                    {+0.657863974571228d, +4.2999935789083046E-8d,}, // 953
                    {+0.6583696603775024d, +4.620570188811826E-8d,}, // 954
                    {+0.6588751077651978d, +3.223791487908353E-8d,}, // 955
                    {+0.659380316734314d, +1.3548138612715822E-9d,}, // 956
                    {+0.6598852872848511d, -4.618575323863973E-8d,}, // 957
                    {+0.6603899002075195d, +9.082960673843353E-9d,}, // 958
                    {+0.6608942747116089d, +4.820873399634487E-8d,}, // 959
                    {+0.6613985300064087d, -4.776104368314602E-8d,}, // 960
                    {+0.6619024276733398d, -4.0151502150238136E-8d,}, // 961
                    {+0.6624060869216919d, -4.791602708710648E-8d,}, // 962
                    {+0.6629093885421753d, +4.8410188461165925E-8d,}, // 963
                    {+0.6634125709533691d, +1.0663697110471944E-8d,}, // 964
                    {+0.6639155149459839d, -4.1691464781797555E-8d,}, // 965
                    {+0.66441810131073d, +1.080835500478704E-8d,}, // 966
                    {+0.664920449256897d, +4.920784622407246E-8d,}, // 967
                    {+0.6654226779937744d, -4.544868396511241E-8d,}, // 968
                    {+0.6659245491027832d, -3.448944157854234E-8d,}, // 969
                    {+0.6664261817932129d, -3.6870882345139385E-8d,}, // 970
                    {+0.6669275760650635d, -5.234055273962444E-8d,}, // 971
                    {+0.6674286127090454d, +3.856291077979099E-8d,}, // 972
                    {+0.6679295301437378d, -2.327375671320742E-9d,}, // 973
                    {+0.6684302091598511d, -5.555080534042001E-8d,}, // 974
                    {+0.6689305305480957d, -1.6471487337453832E-9d,}, // 975
                    {+0.6694306135177612d, +4.042486803683015E-8d,}, // 976
                    {+0.6699305772781372d, -4.8293856891818295E-8d,}, // 977
                    {+0.6704301834106445d, -2.9134931730784303E-8d,}, // 978
                    {+0.6709295511245728d, -2.1058207594753368E-8d,}, // 979
                    {+0.6714286804199219d, -2.3814619551682855E-8d,}, // 980
                    {+0.6719275712966919d, -3.7155475428252136E-8d,}, // 981
                    {+0.6724261045455933d, +5.8376834484391746E-8d,}, // 982
                    {+0.6729245185852051d, +2.4611679969129262E-8d,}, // 983
                    {+0.6734226942062378d, -1.899407107267079E-8d,}, // 984
                    {+0.6739205121994019d, +4.7016079464436395E-8d,}, // 985
                    {+0.6744182109832764d, -1.5529608026276525E-8d,}, // 986
                    {+0.6749155521392822d, +3.203391672602453E-8d,}, // 987
                    {+0.6754127740859985d, -4.8465821804075345E-8d,}, // 988
                    {+0.6759096384048462d, -1.8364507801369988E-8d,}, // 989
                    {+0.6764062643051147d, +3.3739397633046517E-9d,}, // 990
                    {+0.6769026517868042d, +1.6994526063192333E-8d,}, // 991
                    {+0.6773988008499146d, +2.2741891590028428E-8d,}, // 992
                    {+0.6778947114944458d, +2.0860312877435047E-8d,}, // 993
                    {+0.678390383720398d, +1.1593703222523284E-8d,}, // 994
                    {+0.678885817527771d, -4.814386594291911E-9d,}, // 995
                    {+0.6793810129165649d, -2.812076759125914E-8d,}, // 996
                    {+0.6798759698867798d, -5.808261186903479E-8d,}, // 997
                    {+0.680370569229126d, +2.4751837654582522E-8d,}, // 998
                    {+0.6808650493621826d, -1.7793890245755405E-8d,}, // 999
                    {+0.6813591718673706d, +5.294053246347931E-8d,}, // 1000
                    {+0.681853175163269d, -1.2220826223585654E-9d,}, // 1001
                    {+0.6823468208312988d, +5.8377876767612725E-8d,}, // 1002
                    {+0.6828403472900391d, -6.437492120743254E-9d,}, // 1003
                    {+0.6833335161209106d, +4.2990710043633113E-8d,}, // 1004
                    {+0.6838265657424927d, -3.1516131027023284E-8d,}, // 1005
                    {+0.684319257736206d, +8.70017386744679E-9d,}, // 1006
                    {+0.6848117113113403d, +4.466959125843237E-8d,}, // 1007
                    {+0.6853040456771851d, -4.25782656420497E-8d,}, // 1008
                    {+0.6857960224151611d, -1.4386267593671393E-8d,}, // 1009
                    {+0.6862877607345581d, +1.0274494061148778E-8d,}, // 1010
                    {+0.686779260635376d, +3.164186629229597E-8d,}, // 1011
                    {+0.6872705221176147d, +4.995334552140326E-8d,}, // 1012
                    {+0.687761664390564d, -5.3763211240398744E-8d,}, // 1013
                    {+0.6882524490356445d, -4.0852427502515625E-8d,}, // 1014
                    {+0.688742995262146d, -3.0287143914420064E-8d,}, // 1015
                    {+0.6892333030700684d, -2.183125937905008E-8d,}, // 1016
                    {+0.6897233724594116d, -1.524901992178814E-8d,}, // 1017
                    {+0.6902132034301758d, -1.0305018010328949E-8d,}, // 1018
                    {+0.6907027959823608d, -6.764191876212205E-9d,}, // 1019
                    {+0.6911921501159668d, -4.391824838015402E-9d,}, // 1020
                    {+0.6916812658309937d, -2.9535446262017846E-9d,}, // 1021
                    {+0.6921701431274414d, -2.2153227096187463E-9d,}, // 1022
                    {+0.6926587820053101d, -1.943473623641502E-9d,}, // 1023
            };
        }

        /**
         * log(2) (high bits).
         */
        private static final double LN_2_A = 0.693147063255310059;

        /**
         * log(2) (low bits).
         */
        private static final double LN_2_B = 1.17304635250823482e-7;

        private static final long HEX_40000000 = 0x40000000L; // 1073741824L

        /**
         * 2^52 - double numbers this large must be integral (no fraction) or NaN or Infinite
         */
        private static final double TWO_POWER_52 = 4503599627370496.0;

        /**
         * Coefficients for log, when input 0.99 < x < 1.01.
         */
        private static final double LN_QUICK_COEF[][] = {
                {1.0, 5.669184079525E-24},
                {-0.25, -0.25},
                {0.3333333134651184, 1.986821492305628E-8},
                {-0.25, -6.663542893624021E-14},
                {0.19999998807907104, 1.1921056801463227E-8},
                {-0.1666666567325592, -7.800414592973399E-9},
                {0.1428571343421936, 5.650007086920087E-9},
                {-0.12502530217170715, -7.44321345601866E-11},
                {0.11113807559013367, 9.219544613762692E-9},
        };

        /**
         * Natural logarithm.
         *
         * @param x a double
         * @return log(x)
         */
        static final double log(final double x) {
            if (x == 0) { // Handle special case of +0/-0
                return Double.NEGATIVE_INFINITY;
            }
            long bits = Double.doubleToRawLongBits(x);

            /* Handle special cases of negative input, and NaN */
            if ((bits & 0x8000000000000000L) != 0 || x != x) {
                return Double.NaN;
            }

            /* Handle special cases of Positive infinity. */
            if (x == Double.POSITIVE_INFINITY) {
                return Double.POSITIVE_INFINITY;
            }

            /* Extract the exponent */
            int exp = (int) (bits >> 52) - 1023;

            if ((bits & 0x7ff0000000000000L) == 0) {
                /* Normalize the subnormal number. */
                bits <<= 1;
                while ((bits & 0x0010000000000000L) == 0) {
                    --exp;
                    bits <<= 1;
                }
            }


            if ((exp == -1 || exp == 0) && x < 1.01 && x > 0.99) {
                /* The normal method doesn't work well in the range [0.99, 1.01], so call do a straight
                polynomial expansion in higer precision. */

                /* Compute x - 1.0 and split it */
                double xa = x - 1.0;
                double xb;
                double tmp = xa * HEX_40000000;
                double aa = xa + tmp - tmp;
                double ab = xa - aa;
                xa = aa;
                xb = ab;

                final double[] lnCoef_last = LN_QUICK_COEF[LN_QUICK_COEF.length - 1];
                double ya = lnCoef_last[0];
                double yb = lnCoef_last[1];

                for (int i = LN_QUICK_COEF.length - 2; i >= 0; i--) {
                    /* Multiply a = y * x */
                    aa = ya * xa;
                    ab = ya * xb + yb * xa + yb * xb;
                    /* split, so now y = a */
                    tmp = aa * HEX_40000000;
                    ya = aa + tmp - tmp;
                    yb = aa - ya + ab;

                    /* Add  a = y + lnQuickCoef */
                    final double[] lnCoef_i = LN_QUICK_COEF[i];
                    aa = ya + lnCoef_i[0];
                    ab = yb + lnCoef_i[1];
                    /* Split y = a */
                    tmp = aa * HEX_40000000;
                    ya = aa + tmp - tmp;
                    yb = aa - ya + ab;
                }

                /* Multiply a = y * x */
                aa = ya * xa;
                ab = ya * xb + yb * xa + yb * xb;
                /* split, so now y = a */
                tmp = aa * HEX_40000000;
                ya = aa + tmp - tmp;
                yb = aa - ya + ab;

                return ya + yb;
            }

            // lnm is a log of a number in the range of 1.0 - 2.0, so 0 <= lnm < ln(2)
            final double[] lnm = lnMant.LN_MANT[(int) ((bits & 0x000ffc0000000000L) >> 42)];

            // y is the most significant 10 bits of the mantissa
            final double epsilon = (bits & 0x3ffffffffffL) / (TWO_POWER_52 + (bits & 0x000ffc0000000000L));

            double lnza = 0.0;
            double lnzb = 0.0;

            /* High precision not required.  Eval Remez polynomial
            using standard double precision */
            lnza = -0.16624882440418567;
            lnza = lnza * epsilon + 0.19999954120254515;
            lnza = lnza * epsilon + -0.2499999997677497;
            lnza = lnza * epsilon + 0.3333333333332802;
            lnza = lnza * epsilon + -0.5;
            lnza = lnza * epsilon + 1.0;
            lnza *= epsilon;

            /* Relative sizes:
             * lnzb     [0, 2.33E-10]
             * lnm[1]   [0, 1.17E-7]
             * ln2B*exp [0, 1.12E-4]
             * lnza      [0, 9.7E-4]
             * lnm[0]   [0, 0.692]
             * ln2A*exp [0, 709]
             */

            /* Compute the following sum:
             * lnzb + lnm[1] + ln2B*exp + lnza + lnm[0] + ln2A*exp;
             */

            //return lnzb + lnm[1] + ln2B*exp + lnza + lnm[0] + ln2A*exp;
            double a = LN_2_A * exp;
            double b = 0.0;
            double c = a + lnm[0];
            double d = -(c - a - lnm[0]);
            a = c;
            b += d;

            c = a + lnza;
            d = -(c - a - lnza);
            a = c;
            b += d;

            c = a + LN_2_B * exp;
            d = -(c - a - LN_2_B * exp);
            a = c;
            b += d;

            c = a + lnm[1];
            d = -(c - a - lnm[1]);
            a = c;
            b += d;

            c = a + lnzb;
            d = -(c - a - lnzb);
            a = c;
            b += d;

            return a + b;
        }
    }

    /*
     * Copyright 2012-2015 Jeff Hain
     *
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     *     http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     */
    /*
     * =============================================================================
     * Notice of fdlibm package this program is partially derived from:
     *
     * Copyright (C) 1993 by Sun Microsystems, Inc. All rights reserved.
     *
     * Developed at SunSoft, a Sun Microsystems, Inc. business.
     * Permission to use, copy, modify, and distribute this
     * software is freely granted, provided that this notice
     * is preserved.
     * =============================================================================
     */

    /**
     * Math functions from Jafama (2.1.0).
     */
    private static final class JafamaMath {
        static final double TWO_POW_52 = twoPow(52);

        /**
         * Double.MIN_NORMAL since Java 6.
         */
        static final double DOUBLE_MIN_NORMAL = Double.longBitsToDouble(0x0010000000000000L); // 2.2250738585072014E-308

        // Not storing float/double mantissa size in constants,
        // for 23 and 52 are shorter to read and more
        // bitwise-explicit than some constant's name.

        static final int MIN_DOUBLE_EXPONENT = -1074;
        static final int MAX_DOUBLE_EXPONENT = 1023;

        static final double LOG_2 = StrictMath.log(2.0);

        //--------------------------------------------------------------------------
        // CONSTANTS AND TABLES FOR EXP AND EXPM1
        //--------------------------------------------------------------------------

        static final double EXP_OVERFLOW_LIMIT = Double.longBitsToDouble(0x40862E42FEFA39EFL); // 7.09782712893383973096e+02
        static final double EXP_UNDERFLOW_LIMIT = Double.longBitsToDouble(0xC0874910D52D3051L); // -7.45133219101941108420e+02
        static final int EXP_LO_DISTANCE_TO_ZERO_POT = 0;
        static final int EXP_LO_DISTANCE_TO_ZERO = (1 << EXP_LO_DISTANCE_TO_ZERO_POT);
        static final int EXP_LO_TAB_SIZE_POT = getTabSizePower(11);
        static final int EXP_LO_TAB_SIZE = (1 << EXP_LO_TAB_SIZE_POT) + 1;
        static final int EXP_LO_TAB_MID_INDEX = ((EXP_LO_TAB_SIZE - 1) / 2);
        static final int EXP_LO_INDEXING = EXP_LO_TAB_MID_INDEX / EXP_LO_DISTANCE_TO_ZERO;
        static final int EXP_LO_INDEXING_DIV_SHIFT = EXP_LO_TAB_SIZE_POT - 1 - EXP_LO_DISTANCE_TO_ZERO_POT;

        static final class MyTExp {
            static final double[] expHiTab = new double[1 + (int) EXP_OVERFLOW_LIMIT - (int) EXP_UNDERFLOW_LIMIT];
            static final double[] expLoPosTab = new double[EXP_LO_TAB_SIZE];
            static final double[] expLoNegTab = new double[EXP_LO_TAB_SIZE];

            static {
                init();
            }

            private static strictfp void init() {
                for (int i = (int) EXP_UNDERFLOW_LIMIT; i <= (int) EXP_OVERFLOW_LIMIT; i++) {
                    expHiTab[i - (int) EXP_UNDERFLOW_LIMIT] = StrictMath.exp(i);
                }
                for (int i = 0; i < EXP_LO_TAB_SIZE; i++) {
                    // x: in [-EXPM1_DISTANCE_TO_ZERO,EXPM1_DISTANCE_TO_ZERO].
                    double x = -EXP_LO_DISTANCE_TO_ZERO + i / (double) EXP_LO_INDEXING;
                    // exp(x)
                    expLoPosTab[i] = StrictMath.exp(x);
                    // 1-exp(-x), accurately computed
                    expLoNegTab[i] = -StrictMath.expm1(-x);
                }
            }
        }

        //--------------------------------------------------------------------------
        // CONSTANTS AND TABLES FOR LOG AND LOG1P
        //--------------------------------------------------------------------------

        static final int LOG_BITS = getTabSizePower(12);
        static final int LOG_TAB_SIZE = (1 << LOG_BITS);

        static final class MyTLog {
            static final double[] logXLogTab = new double[LOG_TAB_SIZE];
            static final double[] logXTab = new double[LOG_TAB_SIZE];
            static final double[] logXInvTab = new double[LOG_TAB_SIZE];

            static {
                init();
            }

            private static strictfp void init() {
                for (int i = 0; i < LOG_TAB_SIZE; i++) {
                    // Exact to use inverse of tab size, since it is a power of two.
                    double x = 1 + i * (1.0 / LOG_TAB_SIZE);
                    logXLogTab[i] = StrictMath.log(x);
                    logXTab[i] = x;
                    logXInvTab[i] = 1 / x;
                }
            }
        }

        /**
         * @param value A double value.
         * @return e^value.
         */
        static final double exp(double value) {
            // exp(x) = exp([x])*exp(y)
            // with [x] the integer part of x, and y = x-[x]
            // ===>
            // We find an approximation of y, called z.
            // ===>
            // exp(x) = exp([x])*(exp(z)*exp(epsilon))
            // with epsilon = y - z
            // ===>
            // We have exp([x]) and exp(z) pre-computed in tables, we "just" have to compute exp(epsilon).
            //
            // We use the same indexing (cast to int) to compute x integer part and the
            // table index corresponding to z, to avoid two int casts.
            // Also, to optimize index multiplication and division, we use powers of two,
            // so that we can do it with bits shifts.

            if (value > EXP_OVERFLOW_LIMIT) {
                return Double.POSITIVE_INFINITY;
            } else if (!(value >= EXP_UNDERFLOW_LIMIT)) {
                return (value != value) ? Double.NaN : 0.0;
            }

            final int indexes = (int) (value * EXP_LO_INDEXING);

            final int valueInt;
            if (indexes >= 0) {
                valueInt = (indexes >> EXP_LO_INDEXING_DIV_SHIFT);
            } else {
                valueInt = -((-indexes) >> EXP_LO_INDEXING_DIV_SHIFT);
            }
            final double hiTerm = MyTExp.expHiTab[valueInt - (int) EXP_UNDERFLOW_LIMIT];

            final int zIndex = indexes - (valueInt << EXP_LO_INDEXING_DIV_SHIFT);
            final double y = (value - valueInt);
            final double z = zIndex * (1.0 / EXP_LO_INDEXING);
            final double eps = y - z;
            final double expZ = MyTExp.expLoPosTab[zIndex + EXP_LO_TAB_MID_INDEX];
            final double expEps = (1 + eps * (1 + eps * (1.0 / 2 + eps * (1.0 / 6 + eps * (1.0 / 24)))));
            final double loTerm = expZ * expEps;

            return hiTerm * loTerm;
        }

        /**
         * Much more accurate than exp(value)-1,
         * for arguments (and results) close to zero.
         *
         * @param value A double value.
         * @return e^value-1.
         */
        public static double expm1(double value) {
            // If value is far from zero, we use exp(value)-1.
            //
            // If value is close to zero, we use the following formula:
            // exp(value)-1
            // = exp(valueApprox)*exp(epsilon)-1
            // = exp(valueApprox)*(exp(epsilon)-exp(-valueApprox))
            // = exp(valueApprox)*(1+epsilon+epsilon^2/2!+...-exp(-valueApprox))
            // = exp(valueApprox)*((1-exp(-valueApprox))+epsilon+epsilon^2/2!+...)
            // exp(valueApprox) and exp(-valueApprox) being stored in tables.

            if (Math.abs(value) < EXP_LO_DISTANCE_TO_ZERO) {
                // Taking int part instead of rounding, which takes too long.
                int i = (int)(value*EXP_LO_INDEXING);
                double delta = value-i*(1.0/EXP_LO_INDEXING);
                return MyTExp.expLoPosTab[i+EXP_LO_TAB_MID_INDEX]*(MyTExp.expLoNegTab[i+EXP_LO_TAB_MID_INDEX]+delta*(1+delta*(1.0/2+delta*(1.0/6+delta*(1.0/24+delta*(1.0/120))))));
            } else {
                return exp(value)-1;
            }
        }

        /**
         * @param value A double value.
         * @return Value logarithm (base e).
         */
        static final double log(double value) {
            if (value > 0.0) {
                if (value == Double.POSITIVE_INFINITY) {
                    return Double.POSITIVE_INFINITY;
                }

                // For normal values not close to 1.0, we use the following formula:
                // log(value)
                // = log(2^exponent*1.mantissa)
                // = log(2^exponent) + log(1.mantissa)
                // = exponent * log(2) + log(1.mantissa)
                // = exponent * log(2) + log(1.mantissaApprox) + log(1.mantissa/1.mantissaApprox)
                // = exponent * log(2) + log(1.mantissaApprox) + log(1+epsilon)
                // = exponent * log(2) + log(1.mantissaApprox) + epsilon-epsilon^2/2+epsilon^3/3-epsilon^4/4+...
                // with:
                // 1.mantissaApprox <= 1.mantissa,
                // log(1.mantissaApprox) in table,
                // epsilon = (1.mantissa/1.mantissaApprox)-1
                //
                // To avoid bad relative error for small results,
                // values close to 1.0 are treated aside, with the formula:
                // log(x) = z*(2+z^2*((2.0/3)+z^2*((2.0/5))+z^2*((2.0/7))+...)))
                // with z=(x-1)/(x+1)

                double h;
                if (value > 0.95) {
                    if (value < 1.14) {
                        double z = (value - 1.0) / (value + 1.0);
                        double z2 = z * z;
                        return z * (2 + z2 * ((2.0 / 3) + z2 * ((2.0 / 5) + z2 * ((2.0 / 7) + z2 * ((2.0 / 9) + z2 * ((2.0 / 11)))))));
                    }
                    h = 0.0;
                } else if (value < DOUBLE_MIN_NORMAL) {
                    // Ensuring value is normal.
                    value *= TWO_POW_52;
                    // log(x*2^52)
                    // = log(x)-ln(2^52)
                    // = log(x)-52*ln(2)
                    h = -52 * LOG_2;
                } else {
                    h = 0.0;
                }

                int valueBitsHi = (int) (Double.doubleToRawLongBits(value) >> 32);
                int valueExp = (valueBitsHi >> 20) - MAX_DOUBLE_EXPONENT;
                // Getting the first LOG_BITS bits of the mantissa.
                int xIndex = ((valueBitsHi << 12) >>> (32 - LOG_BITS));

                // 1.mantissa/1.mantissaApprox - 1
                double z = (value * twoPowNormalOrSubnormal(-valueExp)) * MyTLog.logXInvTab[xIndex] - 1;

                z *= (1 - z * ((1.0 / 2) - z * ((1.0 / 3))));

                return h + valueExp * LOG_2 + (MyTLog.logXLogTab[xIndex] + z);

            } else if (value == 0.0) {
                return Double.NEGATIVE_INFINITY;
            } else { // value < 0.0, or value is NaN
                return Double.NaN;
            }
        }

        /**
         * Much more accurate than log(1+value),
         * for arguments (and results) close to zero.
         *
         * @param value A double value.
         * @return Logarithm (base e) of (1+value).
         */
        static final double log1p(double value) {
            if (value > -1.0) {
                if (value == Double.POSITIVE_INFINITY) {
                    return Double.POSITIVE_INFINITY;
                }

                // ln'(x) = 1/x
                // so
                // log(x+epsilon) ~= log(x) + epsilon/x
                //
                // Let u be 1+value rounded:
                // 1+value = u+epsilon
                //
                // log(1+value)
                // = log(u+epsilon)
                // ~= log(u) + epsilon/value
                // We compute log(u) as done in log(double), and then add the corrective term.

                double valuePlusOne = 1.0 + value;
                if (valuePlusOne == 1.0) {
                    return value;
                } else if (Math.abs(value) < 0.15) {
                    double z = value / (value + 2.0);
                    double z2 = z * z;
                    return z * (2 + z2 * ((2.0 / 3) + z2 * ((2.0 / 5) + z2 * ((2.0 / 7) + z2 * ((2.0 / 9) + z2 * ((2.0 / 11)))))));
                }

                int valuePlusOneBitsHi = (int) (Double.doubleToRawLongBits(valuePlusOne) >> 32) & 0x7FFFFFFF;
                int valuePlusOneExp = (valuePlusOneBitsHi >> 20) - MAX_DOUBLE_EXPONENT;
                // Getting the first LOG_BITS bits of the mantissa.
                int xIndex = ((valuePlusOneBitsHi << 12) >>> (32 - LOG_BITS));

                // 1.mantissa/1.mantissaApprox - 1
                double z = (valuePlusOne * twoPowNormalOrSubnormal(-valuePlusOneExp)) * MyTLog.logXInvTab[xIndex] - 1;

                z *= (1 - z * ((1.0 / 2) - z * (1.0 / 3)));

                // Adding epsilon/valuePlusOne to z,
                // with
                // epsilon = value - (valuePlusOne-1)
                // (valuePlusOne + epsilon ~= 1+value (not rounded))

                return valuePlusOneExp * LOG_2 + MyTLog.logXLogTab[xIndex] + (z + (value - (valuePlusOne - 1)) / valuePlusOne);
            } else if (value == -1.0) {
                return Double.NEGATIVE_INFINITY;
            } else { // value < -1.0, or value is NaN
                return Double.NaN;
            }
        }

        /**
         * 1e-13ish accuracy or better on whole double range.
         *
         * @param value A double value.
         * @param power A power.
         * @return value^power.
         */
        static final double pow(double value, double power) {
            if (power == 0.0) {
                return 1.0;
            } else if (power == 1.0) {
                return value;
            }
            if (value <= 0.0) {
                // powerInfo: 0 if not integer, 1 if even integer, -1 if odd integer
                int powerInfo;
                if (Math.abs(power) >= (TWO_POW_52 * 2)) {
                    // The binary digit just before comma is outside mantissa,
                    // thus it is always 0: power is an even integer.
                    powerInfo = 1;
                } else {
                    // If power's magnitude permits, we cast into int instead of into long,
                    // as it is faster.
                    if (Math.abs(power) <= (double) Integer.MAX_VALUE) {
                        int powerAsInt = (int) power;
                        if (power == (double) powerAsInt) {
                            powerInfo = ((powerAsInt & 1) == 0) ? 1 : -1;
                        } else { // power is not an integer (and not NaN, due to test against Integer.MAX_VALUE)
                            powerInfo = 0;
                        }
                    } else {
                        long powerAsLong = (long) power;
                        if (power == (double) powerAsLong) {
                            powerInfo = ((powerAsLong & 1) == 0) ? 1 : -1;
                        } else { // power is not an integer, or is NaN
                            if (power != power) {
                                return Double.NaN;
                            }
                            powerInfo = 0;
                        }
                    }
                }

                if (value == 0.0) {
                    if (power < 0.0) {
                        return (powerInfo < 0) ? 1 / value : Double.POSITIVE_INFINITY;
                    } else { // power > 0.0 (0 and NaN cases already treated)
                        return (powerInfo < 0) ? value : 0.0;
                    }
                } else { // value < 0.0
                    if (value == Double.NEGATIVE_INFINITY) {
                        if (powerInfo < 0) { // power odd integer
                            return (power < 0.0) ? -0.0 : Double.NEGATIVE_INFINITY;
                        } else { // power even integer, or not an integer
                            return (power < 0.0) ? 0.0 : Double.POSITIVE_INFINITY;
                        }
                    } else {
                        return (powerInfo == 0) ? Double.NaN : powerInfo * exp(power * log(-value));
                    }
                }
            } else { // value > 0.0, or value is NaN
                return exp(power * log(value));
            }
        }


        //--------------------------------------------------------------------------
        // PACKAGE-PRIVATE METHODS
        //--------------------------------------------------------------------------

        /**
         * @param power Must be in normal or subnormal values range.
         */
        private static final double twoPowNormalOrSubnormal(int power) {
            if (power <= -MAX_DOUBLE_EXPONENT) { // Not normal.
                return Double.longBitsToDouble(0x0008000000000000L >> (-(power + MAX_DOUBLE_EXPONENT)));
            } else { // Normal.
                return Double.longBitsToDouble(((long) (power + MAX_DOUBLE_EXPONENT)) << 52);
            }
        }

        /**
         * Use look-up tables size power through this method,
         * to make sure is it small in case java.lang.Math
         * is directly used.
         */
        private static final int getTabSizePower(int tabSizePower) {
            return tabSizePower;
        }

        /**
         * Returns the exact result, provided it's in double range,
         * i.e. if power is in [-1074,1023].
         *
         * @param power An int power.
         * @return 2^power as a double, or +-Infinity in case of overflow.
         */
        private static final double twoPow(int power) {
            if (power <= -MAX_DOUBLE_EXPONENT) { // Not normal.
                if (power >= MIN_DOUBLE_EXPONENT) { // Subnormal.
                    return Double.longBitsToDouble(0x0008000000000000L >> (-(power + MAX_DOUBLE_EXPONENT)));
                } else { // Underflow.
                    return 0.0;
                }
            } else if (power > MAX_DOUBLE_EXPONENT) { // Overflow.
                return Double.POSITIVE_INFINITY;
            } else { // Normal.
                return Double.longBitsToDouble(((long) (power + MAX_DOUBLE_EXPONENT)) << 52);
            }
        }
    }
}