/* Copyright 2006 aQute SARL 
 * Licensed under the Apache License, Version 2.0, see http://www.apache.org/licenses/LICENSE-2.0 */
package aQute.lib.osgi.eclipse;

import java.io.*;
import java.util.*;
import java.util.regex.*;

import javax.xml.parsers.*;

import org.w3c.dom.*;
import org.xml.sax.*;

import aQute.lib.reporter.*;

/**
 * Parse the Eclipse project information for the classpath. Unfortunately, it is
 * impossible to read the variables. They are ignored but that can cause
 * problems.
 * 
 * @version $Revision: 1.1 $
 */
public class EclipseClasspath {
	static DocumentBuilderFactory	documentBuilderFactory	= DocumentBuilderFactory
																	.newInstance();
	DocumentBuilder					db;
	File							project;
	File							workspace;
	List							sources					= new ArrayList();
	List							classpath				= new ArrayList();
	List							dependents				= new ArrayList();
	File							output;
	boolean							recurse					= true;
	List							exports					= new ArrayList();
	Map								properties				= new HashMap();
	Reporter						reporter;
	int								options;
	List							bootclasspath			= new ArrayList();

	public final static int			DO_VARIABLES			= 1;

	/**
	 * Parse an Eclipse project structure to discover the classpath.
	 * 
	 * @param workspace
	 *            Points to workspace
	 * @param project
	 *            Points to project
	 * @throws ParserConfigurationException
	 * @throws SAXException
	 * @throws IOException
	 */

	public EclipseClasspath(Reporter reporter, File workspace, File project,
			int options) throws Exception {
		this.project = project;
		this.workspace = workspace;
		this.reporter = reporter;
		db = documentBuilderFactory.newDocumentBuilder();
		parse(project, true);
		db = null;
	}

	public EclipseClasspath(Reporter reporter, File workspace, File project)
			throws Exception {
		this(reporter, workspace, project, 0);
	}

	/**
	 * Recursive routine to parse the files. If a sub project is detected, it is
	 * parsed before the parsing continues. This should give the right order.
	 * 
	 * @param project
	 *            Project directory
	 * @param top
	 *            If this is the top project
	 * @throws ParserConfigurationException
	 * @throws SAXException
	 * @throws IOException
	 */
	void parse(File project, boolean top) throws ParserConfigurationException,
			SAXException, IOException {
		File file = new File(project, ".classpath");
		if (!file.exists())
			throw new IllegalArgumentException(".classpath file not found: "
					+ file.getAbsolutePath());

		File cp = new File(project, ".classpath");
		if (!cp.exists())
			return;

		Document doc = db.parse(cp);
		NodeList nodelist = doc.getDocumentElement().getElementsByTagName(
				"classpathentry");

		if (nodelist == null)
			throw new IllegalArgumentException(
					"Can not find classpathentry in classpath file");

		for (int i = 0; i < nodelist.getLength(); i++) {
			Node node = nodelist.item(i);
			NamedNodeMap attrs = node.getAttributes();
			String kind = get(attrs, "kind");
			if ("src".equals(kind)) {
				String path = get(attrs, "path");
				boolean exported = "true".equalsIgnoreCase(get(attrs,
						"exported"));
				if (path.startsWith("/")) {
					// We have another project
					path = path.replace('/', File.separatorChar);
					File subProject = new File(workspace, path.substring(1));
					dependents.add(subProject);
					if (recurse && (top || exported))
						parse(subProject, false);
				} else {
					File src = new File(project, path);
					sources.add(src);
				}
			} else if ("lib".equals(kind)) {
				String path = get(attrs, "path");
				boolean exported = "true".equalsIgnoreCase(get(attrs,
						"exported"));
				if (top || exported) {
					File jar = null;
					path = path.replace('/', File.separatorChar);
					if (path.startsWith(File.separator))
						jar = new File(workspace, path.substring(1));
					else {
						File f = new File(path);
						if (f.isAbsolute()) {
							jar = f;
						} else {
							jar = new File(project, path);
						}
					}
					if ( jar.getName().startsWith("ee."))
						bootclasspath.add(jar);
					else
						classpath.add(jar);
					if (exported)
						exports.add(jar);
				}
			} else if ("output".equals(kind)) {
				String path = get(attrs, "path");
				path = path.replace('/', File.separatorChar);
				output = new File(project, path);
				classpath.add(output);
				exports.add(output);
			} else if ("var".equals(kind)) {
				boolean exported = "true".equalsIgnoreCase(get(attrs,
						"exported"));
				File lib = replaceVar(get(attrs, "path"));
				File slib = replaceVar(get(attrs, "sourcepath"));
				if (lib != null) {
					classpath.add(lib);
					if (exported)
						exports.add(lib);
				}
				if (slib != null)
					sources.add(slib);
			}
		}
	}

	static Pattern	PATH	= Pattern.compile("([A-Z_]+)/(.*)");

	private File replaceVar(String path) {
		if ((options & DO_VARIABLES) == 0)
			return null;

		Matcher m = PATH.matcher(path);
		if (m.matches()) {
			String var = m.group(1);
			String remainder = m.group(2);
			String base = (String) properties.get(var);
			if (base != null) {
				File b = new File(base);
				File f = new File(b, remainder.replace('/', File.separatorChar));
				return f;
			} else
				reporter.error("Can't find replacement variable for: " + path);
		} else
			reporter.error("Cant split variable path: " + path);
		return null;
	}

	private String get(NamedNodeMap map, String name) {
		Node node = map.getNamedItem(name);
		if (node == null)
			return null;

		return node.getNodeValue();
	}

	public List getClasspath() {
		return classpath;
	}

	public List getSourcepath() {
		return sources;
	}

	public File getOutput() {
		return output;
	}

	public List getDependents() {
		return dependents;
	}

	public void setRecurse(boolean recurse) {
		this.recurse = recurse;
	}

	public List getExports() {
		return exports;
	}

	public void setExports(List exports) {
		this.exports = exports;
	}

	public void setProperties(Map map) {
		this.properties = map;
	}

	public List getBootclasspath() {
		return bootclasspath;
	}

}
