/* Copyright 2006 aQute SARL 
 * Licensed under the Apache License, Version 2.0, see http://www.apache.org/licenses/LICENSE-2.0 */

package aQute.bnd.main;

import java.io.*;
import java.util.*;
import java.util.jar.*;
import java.util.zip.*;

import aQute.lib.osgi.*;
import aQute.lib.osgi.eclipse.*;

/**
 * Utility to make bundles.
 * 
 * TODO Add Javadoc comment for this type.
 * 
 * @version $Revision: 1.5 $
 */
public class bnd extends Processor {
	PrintStream		out			= System.out;
	static boolean	exceptions	= false;

	static boolean	failok		= false;

	public static void main(String args[]) {
		bnd main = new bnd();
		try {
			main.run(args);
			if ( main.failok )
				return;
			
			System.exit(main.getErrors().size());
			
		} catch (Exception e) {
			System.err.println("Software error occurred " + e);
			if (exceptions)
				e.printStackTrace();
		}
	}

	void run(String[] args) throws Exception {
		int cnt = 0;
		for (int i = 0; i < args.length; i++) {
			if ("-failok".equals(args[i])) {
				failok = true;
			} else if ("-exceptions".equals(args[i])) {
				exceptions = true;
			} else if ("-pedantic".equals(args[i])) {
				setPedantic(true);
			} else if ("wrap".equals(args[i])) {
				cnt++;
				doWrap(args, ++i);
				break;
			} else if ("print".equals(args[i])) {
				cnt++;
				doPrint(args, ++i);
				break;
			} else if ("view".equals(args[i])) {
				cnt++;
				doView(args, ++i);
				break;
			} else if ("build".equals(args[i])) {
				cnt++;
				doBuild(args, ++i);
				break;
			} else if ("xref".equals(args[i])) {
				cnt++;
				doXref(args, ++i);
				break;
			} else if ("eclipse".equals(args[i])) {
				cnt++;
				doEclipse(args, ++i);
				break;
			} else if ("help".equals(args[i])) {
				cnt++;
				doHelp(args, ++i);
				break;
			} else {
				cnt++;
				String path = args[i];
				if (path.startsWith("-")) {
					doHelp(args, i);
					break;
				} else {
					if (path.endsWith(DEFAULT_BND_EXTENSION))
						doBuild(new File(path), new File[0], new File[0], null,
								"", new File(path).getParentFile(), 0,
								new HashSet());
					else if (path.endsWith(DEFAULT_JAR_EXTENSION)
							|| path.endsWith(DEFAULT_BAR_EXTENSION))
						doPrint(path, -1);
					else {
						doHelp(args, i);
						break;
					}
				}
			}
		}

		if (cnt == 0) {
			File f = new File("bnd.bnd");
			if (f.exists()) {
				doBuild(f, new File[0], new File[0], null, "", f
						.getParentFile(), 0, new HashSet());
			} else {
				doHelp();
			}
		}
		int n = 1;
		switch (getErrors().size()) {
		case 0:
			// System.err.println("No errors");
			break;
		case 1:
			System.err.println("One error");
			break;
		default:
			System.err.println(getErrors().size() + " errors");
		}
		for (Iterator i = getErrors().iterator(); i.hasNext();) {
			String msg = (String) i.next();
			System.err.println(n++ + " : " + msg);
		}
		n = 1;
		switch (getWarnings().size()) {
		case 0:
			// System.err.println("No warnings");
			break;
		case 1:
			System.err.println("One warning");
			break;
		default:
			System.err.println(getErrors().size() + " warnings");
		}
		for (Iterator i = getWarnings().iterator(); i.hasNext();) {
			String msg = (String) i.next();
			System.err.println(n++ + " : " + msg);
		}

	}

	/**
	 * Cross reference every class in the jar file to the files it references
	 * 
	 * @param args
	 * @param i
	 */

	private void doXref(String[] args, int i) {
		for (; i < args.length; i++) {
			try {
				File file = new File(args[i]);
				Jar jar = new Jar(file.getName(), file);
				for (Iterator it = jar.getResources().entrySet().iterator(); it
						.hasNext();) {
					Map.Entry entry = (Map.Entry) it.next();
					String key = (String) entry.getKey();
					Resource r = (Resource) entry.getValue();
					if (key.endsWith(".class")) {
						InputStream in = r.openInputStream();
						Clazz clazz = new Clazz(key);
						out.print(key);
						Set xref = clazz.xref(in);
						in.close();
						for (Iterator x = xref.iterator(); x.hasNext();) {
							out.print("\t");
							out.print(x.next());
						}
						out.println();
					}
				}
				jar.close();
			} catch (Exception e) {
				e.printStackTrace();
			}
		}

	}

	private void doEclipse(String[] args, int i) throws Exception {
		File dir = new File("").getAbsoluteFile();
		if (args.length == i)
			doEclipse(dir);
		else {
			for (; i < args.length; i++) {
				doEclipse(new File(dir, args[i]));
			}
		}
	}

	private void doEclipse(File file) throws Exception {
		if (!file.isDirectory())
			error("Eclipse requires a path to a directory: "
					+ file.getAbsolutePath());
		else {
			File cp = new File(file, ".classpath");
			if (!cp.exists()) {
				error("Cannot find .classpath in project directory: "
						+ file.getAbsolutePath());
			} else {
				EclipseClasspath eclipse = new EclipseClasspath(this, file
						.getParentFile(), file);
				out.printf("Classpath    %s", new Object[] { eclipse
						.getClasspath() });
				out.println();
				out.printf("Dependents   %s", new Object[] { eclipse
						.getDependents() });
				out.println();
				out.printf("Sourcepath   %s", new Object[] { eclipse
						.getSourcepath() });
				out.println();
				out.printf("Output       %s", new Object[] { eclipse
						.getOutput() });
				out.println();
			}
		}

	}

	final static int	BUILD_SOURCES	= 1;
	final static int	BUILD_POM		= 2;
	final static int	BUILD_FORCE		= 4;

	private void doBuild(String[] args, int i) throws Exception {
		File classpath[] = new File[0];
		File workspace = null;
		File sourcepath[] = new File[0];
		File output = null;
		String eclipse = "";
		int options = 0;

		for (; i < args.length; i++) {
			if ("-workspace".startsWith(args[i])) {
				workspace = new File(args[++i]);
			} else if ("-classpath".startsWith(args[i])) {
				classpath = getClasspath(args[++i]);
			} else if ("-sourcepath".startsWith(args[i])) {
				String arg = args[++i];
				String spaces[] = arg.split("\\s*,\\s*");
				sourcepath = new File[spaces.length];
				for (int j = 0; j < spaces.length; j++) {
					File f = new File(spaces[j]);
					if (!f.exists())
						error("No such sourcepath entry: "
								+ f.getAbsolutePath());
					sourcepath[j] = f;
				}
			} else if ("-eclipse".startsWith(args[i])) {
				eclipse = args[++i];
			} else if ("-noeclipse".startsWith(args[i])) {
				eclipse = null;
			} else if ("-output".startsWith(args[i])) {
				output = new File(args[++i]);
			} else if ("-sources".startsWith(args[i])) {
				options |= BUILD_SOURCES;
			} else if ("-pom".startsWith(args[i])) {
				options |= BUILD_POM;
			} else if ("-force".startsWith(args[i])) {
				options |= BUILD_FORCE;
			} else {
				File properties = new File(args[i]);

				if (!properties.exists())
					error("Cannot find properties file: " + args[i]);
				else {
					if (workspace == null)
						workspace = properties.getParentFile();

					doBuild(properties, classpath, sourcepath, output, eclipse,
							workspace, options, new HashSet());
				}
				output = null;
			}
		}

	}

	private File[] getClasspath(String string) {
		String spaces[] = string.split("\\s*,\\s*");
		File classpath[] = new File[spaces.length];
		for (int j = 0; j < spaces.length; j++) {
			File f = new File(spaces[j]);
			if (!f.exists())
				error("No such classpath entry: " + f.getAbsolutePath());
			classpath[j] = f;
		}
		return classpath;
	}

	private void doBuild(File properties, File classpath[], File sourcepath[],
			File output, String eclipse, File workspace, int options,
			Set building) throws Exception {

		properties = properties.getAbsoluteFile();
		if (building.contains(properties)) {
			error("Circular dependency in pre build " + properties);
			return;
		}
		building.add(properties);

		if (output == null) {
			output = properties.getAbsoluteFile().getParentFile();
		}

		Builder builder = new Builder();
		builder.setPedantic(isPedantic());
		builder.setProperties(properties);

		String prebuild = builder.getProperty("-prebuild");
		if (prebuild != null)
			prebuild(prebuild, properties.getParentFile(), classpath,
					sourcepath, output, eclipse, workspace, options, building);

		doEclipse(builder, properties, classpath, sourcepath, eclipse,
				workspace);

		builder.setClasspath(classpath);

		if ((options & BUILD_SOURCES) != 0)
			builder.getProperties().setProperty("-sources", "true");

		Jar jar = builder.build();
		getInfo(builder);
		if (getErrors().size() > 0 && !failok)
			return;

		String name = properties.getName();
		int n = name.lastIndexOf('.');
		if (n > 0)
			name = name.substring(0, n) + DEFAULT_JAR_EXTENSION;
		else
			name = name + DEFAULT_JAR_EXTENSION;

		if (output.isDirectory())
			output = new File(output, name);

		if ((options & BUILD_POM) != 0)
			builder.doPom(jar);

		jar.setName(output.getName());

		String msg = "";
		if (!output.exists() || output.lastModified() <= jar.lastModified()
				|| (options & BUILD_FORCE) != 0) {
			jar.write(output);
		} else {
			msg = "(not modified)";
		}
		statistics(jar, output, msg);
		builder.close();
	}

	private void prebuild(String prebuild, File base, File[] classpath,
			File[] sourcepath, File output, String eclipse2, File workspace,
			int options, Set building) throws Exception {

		// Force the output a directory
		if (output.isFile())
			output = output.getParentFile();

		String parts[] = prebuild.split("\\s*,\\s*");
		for (int i = 0; i < parts.length; i++) {
			File f = new File(parts[i]);
			if (!f.exists())
				f = new File(base, parts[i]);
			if (!f.exists()) {
				error("Trying to build a non-existent file: " + parts[i]);
				continue;
			}
			try {
				doBuild(f, classpath, sourcepath, output, eclipse2, workspace,
						options, building);
			} catch (Exception e) {
				error("Trying to build: " + parts[i] + " " + e);
			}
		}
	}

	private void statistics(Jar jar, File output, String msg) {
		out.printf("%-20s %10d resources %12d bytes %-16s", new Object[] {
				jar.getName(), new Integer(jar.getResources().size()),
				new Integer((int) output.length()), msg });
		out.println();
	}

	/**
	 * @param properties
	 * @param classpath
	 * @param eclipse
	 * @return
	 * @throws IOException
	 */
	void doEclipse(Builder builder, File properties, File[] classpath,
			File sourcepath[], String eclipse, File workspace)
			throws IOException {
		if (eclipse != null) {
			File project = new File(workspace, eclipse).getAbsoluteFile();
			if (project.exists() && project.isDirectory()) {
				try {

					EclipseClasspath path = new EclipseClasspath(this, project
							.getParentFile(), project);
					List newClasspath = new ArrayList(Arrays.asList(classpath));
					newClasspath.addAll(path.getClasspath());
					classpath = (File[]) newClasspath.toArray(classpath);

					List newSourcepath = new ArrayList(Arrays
							.asList(sourcepath));
					newSourcepath.addAll(path.getSourcepath());
					sourcepath = (File[]) newSourcepath.toArray(sourcepath);
				} catch (Exception e) {
					if (eclipse.length() > 0)
						error("Eclipse specified (" + eclipse
								+ ") but getting error processing: " + e);
				}
			} else {
				if (eclipse.length() > 0)
					error("Eclipse specified (" + eclipse
							+ ") but no project directory found");
			}
		}
		builder.setClasspath(classpath);
		builder.setSourcepath(sourcepath);
	}

	private void doHelp() {
		doHelp(new String[0], 0);
	}

	private void doHelp(String[] args, int i) {
		if (args.length <= i) {
			System.out
					.println("bnd -failok? -exceptions? ( wrap | print | build | eclipse | xref | view )?");
			System.out.println("See http://www.aQute.biz/Code/Bnd");
		} else {
			while (args.length > i) {
				if ("wrap".equals(args[i])) {
					System.out
							.println("bnd wrap (-output <file|dir>)? (-properties <file>)? <jar-file>");
				} else if ("print".equals(args[i])) {
					System.out
							.println("bnd wrap -verify? -manifest? -list? -eclipse <jar-file>");
				} else if ("build".equals(args[i])) {
					System.out
							.println("bnd build (-output <file|dir>)? (-classpath <list>)? (-sourcepath <list>)? ");
					System.out
							.println("    -eclipse? -noeclipse? -sources? <bnd-file>");
				} else if ("eclipse".equals(args[i])) {
					System.out.println("bnd eclipse");
				} else if ("view".equals(args[i])) {
					System.out.println("bnd view <file.jar> <resource-names>+");
				}
				i++;
			}
		}
	}

	final static int	VERIFY		= 1;

	final static int	MANIFEST	= 2;

	final static int	LIST		= 4;

	final static int	ECLIPSE		= 8;
	final static int	IMPEXP		= 16;
	final static int	USES		= 32;
	final static int	USEDBY		= 64;

	static final int	HEX			= 0;

	private void doPrint(String[] args, int i) throws Exception {
		int options = 0;

		for (; i < args.length; i++) {
			if ("-verify".startsWith(args[i]))
				options |= VERIFY;
			else if ("-manifest".startsWith(args[i]))
				options |= MANIFEST;
			else if ("-list".startsWith(args[i]))
				options |= LIST;
			else if ("-eclipse".startsWith(args[i]))
				options |= ECLIPSE;
			else if ("-impexp".startsWith(args[i]))
				options |= IMPEXP;
			else if ("-uses".startsWith(args[i]))
				options |= USES;
			else if ("-usedby".startsWith(args[i]))
				options |= USEDBY;
			else if ("-all".startsWith(args[i]))
				options = -1;
			else
				doPrint(args[i], options);
		}
	}

	private void doPrint(String string, int options) throws Exception {
		File file = new File(string);
		if (!file.exists())
			error("File to print not found: " + string);
		else {
			if (options == 0)
				options = VERIFY | MANIFEST | IMPEXP | USES;

			Jar jar = new Jar(file.getName(), file);
			if ((options & VERIFY) != 0) {
				Verifier verifier = new Verifier(jar);
				verifier.setPedantic(isPedantic());
				verifier.verify();
				getInfo(verifier);
			}
			if ((options & MANIFEST) != 0) {
				Manifest manifest = jar.getManifest();
				if (manifest == null)
					warning("JAR file has no manifest " + string);
				else {
					out.println("[MANIFEST " + jar.getName() + "]");
					Set sorted = new TreeSet();
					for (Iterator x = manifest.getMainAttributes().keySet()
							.iterator(); x.hasNext();) {
						Object element = (Object) x.next();
						sorted.add(element.toString());
					}
					for (Iterator e = sorted.iterator(); e.hasNext();) {
						Object key = e.next();
						Object value = manifest.getMainAttributes().getValue(
								(String) key);
						format("%-28s %-50s", new Object[] { key, value });
						out.println();
					}
				}
				out.println();
			}
			if ((options & IMPEXP) != 0) {
				out.println("[IMPEXP]");
				Manifest m = jar.getManifest();
				Map imports = parseHeader(m.getMainAttributes().getValue(
						Analyzer.IMPORT_PACKAGE));
				Map exports = parseHeader(m.getMainAttributes().getValue(
						Analyzer.EXPORT_PACKAGE));
				imports.keySet().removeAll(exports.keySet());
				print("Import-Package", new TreeMap(imports));
				print("Export-Package", new TreeMap(exports));
			}
			if ((options & (USES | USEDBY)) != 0) {
				Analyzer analyzer = new Analyzer();
				analyzer.setPedantic(isPedantic());
				analyzer.setJar(jar);
				analyzer.analyze();
				if ((options & USES) != 0) {
					out.println("[USES]");
					printMapOfSets(new TreeMap(analyzer.getUses()));
				}
				if ((options & USEDBY) != 0) {
					out.println("[USEDBY]");
					printMapOfSets(invertMapOfCollection(analyzer.getUses()));
				}
			}
			if ((options & LIST) != 0) {
				out.println("[LIST]");
				for (Iterator i = jar.getDirectories().entrySet().iterator(); i
						.hasNext();) {
					Map.Entry entry = (Map.Entry) i.next();
					String name = (String) entry.getKey();
					Map contents = (Map) entry.getValue();
					out.println(name);
					for (Iterator c = contents.keySet().iterator(); c.hasNext();) {
						String element = (String) c.next();
						int n = element.lastIndexOf('/');
						if (n > 0)
							element = element.substring(n + 1);
						out.print("  ");
						out.print(element);
						out.println();
					}
				}
				out.println();
			}
			jar.close();
		}
	}

	Map invertMapOfCollection(Map map) {
		Map result = new TreeMap();
		for (Iterator i = map.entrySet().iterator(); i.hasNext();) {
			Map.Entry entry = (Map.Entry) i.next();
			String name = (String) entry.getKey();
			if (name.startsWith("java.") && !name.equals("java.sql"))
				continue;
			Collection used = (Collection) entry.getValue();
			for (Iterator k = used.iterator(); k.hasNext();) {
				String n = (String) k.next();
				if (n.startsWith("java.") && !n.equals("java.sql"))
					continue;
				Set set = (Set) result.get(n);
				if (set == null) {
					set = new TreeSet();
					result.put(n, set);
				}
				set.add(name);
			}
		}
		return result;
	}

	void printMapOfSets(Map map) {
		for (Iterator i = map.entrySet().iterator(); i.hasNext();) {
			Map.Entry entry = (Map.Entry) i.next();
			String name = (String) entry.getKey();
			Collection used = new TreeSet((Collection) entry.getValue());
			for (Iterator k = used.iterator(); k.hasNext();) {
				String n = (String) k.next();
				if (n.startsWith("java.") && !n.equals("java.sql"))
					k.remove();
			}
			String list = vertical(40, used);
			format("%-40s %s", new Object[] { name, list });
		}
	}

	String vertical(int padding, Collection used) {
		StringBuffer sb = new StringBuffer();
		String del = "";
		for (Iterator u = used.iterator(); u.hasNext();) {
			String name = (String) u.next();
			sb.append(del);
			sb.append(name);
			sb.append("\r\n");
			del = pad(padding);
		}
		return sb.toString();
	}

	String pad(int i) {
		StringBuffer sb = new StringBuffer();
		while (i-- > 0)
			sb.append(' ');
		return sb.toString();
	}

	/**
	 * View files from JARs
	 * 
	 * We parse the commandline and print each file on it.
	 * 
	 * @param args
	 * @param i
	 * @throws Exception
	 */
	private void doView(String[] args, int i) throws Exception {
		int options = 0;
		String charset = "UTF-8";

		for (; i < args.length; i++) {
			if ("-charset".startsWith(args[i]))
				charset = args[++i];
			else
				break;
		}

		if (i >= args.length) {
			System.err.println("Insufficient arguments for view, no JAR file");
			return;
		}
		String jar = args[i++];
		if (i >= args.length) {
			System.err.println("No Files to view");
			return;
		}

		File path = new File(jar).getAbsoluteFile();
		if (path.exists()) {
			doView(path, args, i, charset, options);
		} else {
			doView(path, args, i, charset, options);
		}
	}

	private void doView(File path, String[] args, int i, String charset,
			int options) {
		File dir = path.getParentFile();
		if (dir == null) {
			dir = new File("");
		}
		String name = path.getName();
		if (name == null)
			name = "META-INF/MANIFEST.MF";

		Instruction instruction = Instruction.getPattern(path.getName());

		File[] children = dir.listFiles();
		for (int j = 0; j < children.length; j++) {
			String base = children[j].getName();
			// System.out.println("Considering: " +
			// children[j].getAbsolutePath() + " " +
			// instruction.getPattern());
			if (instruction.matches(base) ^ instruction.isNegated()) {
				for (; i < args.length; i++) {
					doView(children[j], args[i], charset, options);
				}
			}
		}
	}

	private void doView(File file, String resource, String charset, int options) {
		// System.out.println("doView:" + file.getAbsolutePath() );
		try {
			Instruction instruction = Instruction.getPattern(resource);
			FileInputStream fin = new FileInputStream(file);
			ZipInputStream in = new ZipInputStream(fin);
			ZipEntry entry = in.getNextEntry();
			while (entry != null) {
				// System.out.println(" Resource: " + entry.getName() + " " +
				// instruction.getPattern() );
				if (instruction.matches(entry.getName())
						^ instruction.isNegated())
					doView(in, charset, options);
				entry = in.getNextEntry();
			}
			in.close();
			fin.close();
		} catch (Exception e) {
			System.out.println("Can not process: " + file.getAbsolutePath());
			e.printStackTrace();
		}
	}

	private void doView(ZipInputStream in, String charset, int options)
			throws Exception {
		// System.out.println("processing");
		InputStreamReader rds = new InputStreamReader(in, charset);
		OutputStreamWriter wrt = new OutputStreamWriter(System.out);
		copy(rds, wrt);
		// rds.close(); also closes the stream which closes our zip file it
		// seems
		wrt.close();
		rds.close();
	}

	private void copy(InputStreamReader rds, OutputStreamWriter wrt)
			throws IOException {
		char buffer[] = new char[1024];
		int size = rds.read(buffer);
		while (size > 0) {
			wrt.write(buffer, 0, size);
			size = rds.read(buffer);
		}
	}

	private void print(String msg, Map ports) {
		if (ports.isEmpty())
			return;
		out.println(msg);
		for (Iterator i = ports.entrySet().iterator(); i.hasNext();) {
			Map.Entry entry = (Map.Entry) i.next();
			String key = (String) entry.getKey();
			Map clause = new TreeMap((Map) entry.getValue());
			clause.remove("uses:");
			format("  %-28s %-40s\r\n", new Object[] { key.trim(),
					clause.isEmpty() ? "" : clause.toString() });
		}
	}

	private void format(String string, Object[] objects) {
		StringBuffer sb = new StringBuffer();
		int index = 0;
		for (int i = 0; i < string.length(); i++) {
			char c = string.charAt(i);
			switch (c) {
			case '%':
				String s = objects[index++] + "";
				int width = 0;
				int justify = -1;

				i++;

				c = string.charAt(i++);
				switch (c) {
				case '-':
					justify = -1;
					break;
				case '+':
					justify = 1;
					break;
				case '|':
					justify = 0;
					break;
				default:
					--i;
				}
				c = string.charAt(i++);
				while (c >= '0' && c <= '9') {
					width *= 10;
					width += c - '0';
					c = string.charAt(i++);
				}
				if (c != 's') {
					throw new IllegalArgumentException(
							"Invalid sprintf format:  " + string);
				}

				if (s.length() > width)
					sb.append(s);
				else {
					switch (justify) {
					case -1:
						sb.append(s);
						for (int j = 0; j < width - s.length(); j++)
							sb.append(" ");
						break;

					case 1:
						for (int j = 0; j < width - s.length(); j++)
							sb.append(" ");
						sb.append(s);
						break;

					case 0:
						int spaces = (width - s.length()) / 2;
						for (int j = 0; j < spaces; j++)
							sb.append(" ");
						sb.append(s);
						for (int j = 0; j < width - s.length() - spaces; j++)
							sb.append(" ");
						break;
					}
				}
				break;

			default:
				sb.append(c);
			}
		}
		out.print(sb);
	}

	private void doWrap(String[] args, int i) throws Exception {
		int options = 0;
		File properties = null;
		File output = null;
		File classpath[] = null;
		for (; i < args.length; i++) {
			if ("-output".startsWith(args[i]))
				output = new File(args[++i]);
			else if ("-properties".startsWith(args[i]))
				properties = new File(args[++i]);
			else if ("-classpath".startsWith(args[i])) {
				classpath = getClasspath(args[++i]);
			} else {
				File bundle = new File(args[i]);
				doWrap(properties, bundle, output, classpath, options, null);
			}
		}
	}

	public boolean doWrap(File properties, File bundle, File output,
			File classpath[], int options, Map additional) throws Exception {
		if (!bundle.exists()) {
			error("No such file: " + bundle.getAbsolutePath());
			return false;
		} else {
			Analyzer analyzer = new Analyzer();
			analyzer.setPedantic(isPedantic());
			analyzer.setJar(bundle);
			Jar dot = analyzer.getJar();

			if (properties != null) {
				analyzer.setProperties(properties);
			}
			if (additional != null)
				analyzer.putAll(additional, false);

			if (analyzer.getProperty(Analyzer.IMPORT_PACKAGE) == null)
				analyzer.setProperty(Analyzer.IMPORT_PACKAGE,
						"*;resolution:=optional");

			if (analyzer.getProperty(Analyzer.EXPORT_PACKAGE) == null) {
				String export = analyzer.calculateExportsFromContents(dot);
				analyzer.setProperty(Analyzer.EXPORT_PACKAGE, export);
			}

			if (classpath != null)
				analyzer.setClasspath(classpath);

			analyzer.mergeManifest(dot.getManifest());

			if (output == null)
				output = properties.getAbsoluteFile().getParentFile();

			String path = bundle.getName() + "$";
			if (path.endsWith(DEFAULT_JAR_EXTENSION + "$"))
				path = path.replace(DEFAULT_JAR_EXTENSION + "$",
						DEFAULT_BAR_EXTENSION);
			else
				path = bundle.getName() + DEFAULT_BAR_EXTENSION;

			if (output.isDirectory())
				output = new File(output, path);

			analyzer.calcManifest();
			Jar jar = analyzer.getJar();
			jar.write(output);
			getInfo(analyzer);
			statistics(jar, output, "");
			analyzer.close();
			return getErrors().size() == 0;
		}
	}

}
