/*
 * Decompiled with CFR 0.152.
 */
package co.instil.dropwizard.assets;

import com.google.common.base.CharMatcher;
import com.google.common.base.Preconditions;
import com.google.common.hash.Hashing;
import com.google.common.io.Files;
import com.google.common.io.Resources;
import com.google.common.net.MediaType;
import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.Charset;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DevelopmentAssetServlet
extends HttpServlet {
    private static final Logger log = LoggerFactory.getLogger(DevelopmentAssetServlet.class);
    private static final long serialVersionUID = 6393345594784987908L;
    private static final CharMatcher SLASHES = CharMatcher.is((char)'/');
    private static final MediaType DEFAULT_MEDIA_TYPE = MediaType.HTML_UTF_8;
    private final String resourcePath;
    private final String uriPath;
    private final String indexFile;
    private final transient Charset defaultCharset;

    public DevelopmentAssetServlet(String resourcePath, String uriPath, String indexFile, Charset defaultCharset) {
        String trimmedPath = SLASHES.trimFrom((CharSequence)resourcePath);
        this.resourcePath = trimmedPath.isEmpty() ? trimmedPath : trimmedPath + '/';
        String trimmedUri = SLASHES.trimTrailingFrom((CharSequence)uriPath);
        this.uriPath = trimmedUri.isEmpty() ? "/" : trimmedUri;
        this.indexFile = indexFile;
        this.defaultCharset = defaultCharset;
    }

    public URL getResourceURL() {
        return Resources.getResource((String)this.resourcePath);
    }

    public String getUriPath() {
        return this.uriPath;
    }

    public String getIndexFile() {
        return this.indexFile;
    }

    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        try {
            CachedAsset cachedAsset;
            StringBuilder builder = new StringBuilder(req.getServletPath());
            if (req.getPathInfo() != null) {
                builder.append(req.getPathInfo());
            }
            if ((cachedAsset = this.loadAsset(builder.toString())) == null) {
                resp.sendError(404);
                return;
            }
            if (this.isCachedClientSide(req, cachedAsset)) {
                resp.sendError(304);
                return;
            }
            resp.setDateHeader("Last-Modified", cachedAsset.getLastModifiedTime());
            resp.setHeader("ETag", cachedAsset.getETag());
            String mimeTypeOfExtension = req.getServletContext().getMimeType(req.getRequestURI());
            MediaType mediaType = DEFAULT_MEDIA_TYPE;
            if (mimeTypeOfExtension != null) {
                try {
                    mediaType = MediaType.parse((String)mimeTypeOfExtension);
                    if (this.defaultCharset != null && mediaType.is(MediaType.ANY_TEXT_TYPE)) {
                        mediaType = mediaType.withCharset(this.defaultCharset);
                    }
                }
                catch (IllegalArgumentException ignore) {
                    log.debug("Ignored exception: {}", (Throwable)ignore);
                }
            }
            resp.setContentType(mediaType.type() + '/' + mediaType.subtype());
            if (mediaType.charset().isPresent()) {
                resp.setCharacterEncoding(((Charset)mediaType.charset().get()).toString());
            }
            try (ServletOutputStream output = resp.getOutputStream();){
                output.write(cachedAsset.getResource());
            }
        }
        catch (RuntimeException | URISyntaxException ignored) {
            resp.sendError(404);
        }
    }

    private CachedAsset loadAsset(String key) throws URISyntaxException, IOException {
        long lastModified;
        Preconditions.checkArgument((boolean)key.startsWith(this.uriPath));
        String requestedResourcePath = SLASHES.trimFrom((CharSequence)key.substring(this.uriPath.length()));
        String absoluteRequestedResourcePath = SLASHES.trimFrom((CharSequence)(this.resourcePath + requestedResourcePath));
        log.debug("Path: {}", (Object)absoluteRequestedResourcePath);
        File requestedResourceURL = new File(absoluteRequestedResourcePath);
        if (requestedResourceURL.isDirectory()) {
            if (this.indexFile != null) {
                requestedResourceURL = new File(absoluteRequestedResourcePath + '/' + this.indexFile);
            } else {
                return null;
            }
        }
        if ((lastModified = requestedResourceURL.lastModified()) < 1L) {
            lastModified = System.currentTimeMillis();
        }
        lastModified = lastModified / 1000L * 1000L;
        return new CachedAsset(Files.toByteArray((File)requestedResourceURL), lastModified);
    }

    private boolean isCachedClientSide(HttpServletRequest req, CachedAsset cachedAsset) {
        return cachedAsset.getETag().equals(req.getHeader("If-None-Match")) || req.getDateHeader("If-Modified-Since") >= cachedAsset.getLastModifiedTime();
    }

    private static class CachedAsset {
        private final byte[] resource;
        private final String eTag;
        private final long lastModifiedTime;

        private CachedAsset(byte[] resource, long lastModifiedTime) {
            this.resource = resource;
            this.eTag = '\"' + Hashing.murmur3_128().hashBytes(resource).toString() + '\"';
            this.lastModifiedTime = lastModifiedTime;
        }

        public byte[] getResource() {
            return this.resource;
        }

        public String getETag() {
            return this.eTag;
        }

        public long getLastModifiedTime() {
            return this.lastModifiedTime;
        }
    }
}

