package be.insaneprogramming.gradle.animalsniffer

import groovy.transform.Canonical
import org.codehaus.mojo.animal_sniffer.ClassListBuilder
import org.codehaus.mojo.animal_sniffer.SignatureChecker
import org.codehaus.mojo.animal_sniffer.logging.Logger
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.slf4j.LoggerFactory

class AnimalSnifferPlugin implements Plugin<Project> {
    private org.slf4j.Logger logger = LoggerFactory.getLogger(this.class)

    @Override
    void apply(Project project) {
        project.configurations.maybeCreate("signature")
        project.extensions.create("animalsniffer", AnimalSnifferExtension)
        if(project.plugins.findPlugin("java")) {
            project.tasks.getByName("compileJava").doLast {
                performAnimalSniffer(project)
            }
        } else {
            logger.warn("Animal Sniffer plugin applied, but Java plugin is not applied. Skipping.")
        }
    }

    def performAnimalSniffer(Project project) {
        if (!project.animalsniffer.skip) {
            if (!project.animalsniffer.signature) {
                throw new IllegalStateException("Signature is required when using Animal Sniffer plugin")
            } else {
                project.dependencies {
                    signature project.animalsniffer.signature
                }
            }
            def logger = new GradleLogger(logger)
            def signatures = project.configurations.signature.resolvedConfiguration.resolvedArtifacts*.file
            ClassListBuilder plb = new ClassListBuilder(logger);
            plb.process(project.buildDir);
            if (project.animalsniffer.excludeDependencies) {
                def files = project.configurations.runtime.resolvedConfiguration.resolvedArtifacts*.file
                files.each {
                    plb.process(it)
                }
            }
            def ignored = plb.getPackages();
            if (project.animalsniffer.ignores) {
                project.animalsniffer.ignores.each {
                    if (it) {
                        ignored.add(it.replace('.', '/'))
                    }
                }
            }
            signatures.each {
                def checker = new SignatureChecker(it.newInputStream(), ignored, logger)
                if (project.animalsniffer.annotations) {
                    checker.setAnnotationTypes(project.animalsniffer.annotations)
                }
                def allSources = project.sourceSets*.allJava.collect { it.srcDirs }.flatten()
                checker.sourcePath = allSources
                if (project.buildDir)
                    checker?.process(project.buildDir)
                if (checker.signatureBroken)
                    throw new IllegalStateException("Signature errors found. Verify them and ignore them with the proper annotation if needed.")
            }
        }
    }
}


@Canonical
class GradleLogger implements Logger {
    @Delegate
    org.slf4j.Logger logger
}

class AnimalSnifferExtension {
    boolean excludeDependencies = true
    String signature = ""
    def ignores = []
    def annotations = []
    boolean skip = false
}

