package be.cylab.mongomail.server;

import be.cylab.mongomail.bizz.Connection;
import java.io.IOException;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.apache.logging.log4j.LogManager;
import be.cylab.mongomail.db.IMailDao;
import java.net.SocketException;

/**
 * Server. Configure the server and launch the socket listener.
 *
 * @author Bunyamin Aslan
 */
public class Server {

    private static ServerSocket server;
    private static boolean running;

    private Server() {

    }

    /**
     * Create a server set by .env file settings and the server is listening
     * sockets.
     *
     * @param dotenv_path relative path to .env
     * @param log4j_path relative path to log4j directory
     */
    public static void launchServer(final String dotenv_path,
            final String log4j_path) {

        System.setProperty(
                "log4j.configurationFile", log4j_path + "log4j2.xml");

        Configuration config = new Configuration(dotenv_path);
        int port = config.getInt("MONGOMAIL_SERVER_PORT");
        String hostname = config.getString("MONGOMAIL_SERVER_HOSTNAME");
        //int core_connections = config.getInt("CONNECTION_CORE");
        int max_connections = config.getInt("MONGOMAIL_CONNECTION_MAX");
        int idle_timeout = config.getInt("MONGOMAIL_IDLE_TIMEOUT");
        IMailDao db = (IMailDao) config.
                getImplementingInstance(IMailDao.class);

        ThreadPoolExecutor thread_pool = new ThreadPoolExecutor(
                /*core_connections*/0, max_connections, idle_timeout,
                TimeUnit.SECONDS, new SynchronousQueue<>());
        running = true;
        Thread t = new Thread() {
            @Override
            public void run() {
                try {
                    server = new ServerSocket(port);
                    LogManager.getLogger(Server.class).info(
                            "Smtp Server " + hostname
                            + " listening on port " + port);
                    while (running) {
                        final Socket socket = server.accept();
                        Connection connection = new Connection(socket, db);
                        thread_pool.submit(connection);
                        LogManager.getLogger(Server.class).info(
                                "New connection from "
                                + socket.getInetAddress().getHostAddress());
                    }
                } catch (IOException exc) {
                    //Socket Exception is thrown only on server.close()
                    if (exc.getClass().equals(SocketException.class)) {
                        LogManager.getLogger(Server.class).
                                fatal("Server successfully closed");
                    } else { //Other unhandled exceptions
                        LogManager.getLogger(Server.class).
                                fatal("Unable to open serverSocket");
                    }
                }
            }

        };
        t.start();

    }

    /**
     * Stop the server properly.
     */
    public static void stopServer() {
        try {
            running = false;
            server.close();
        } catch (IOException ex) {
            LogManager.getLogger(Server.class).
                    fatal("Unable to close serverSocket");
        }
    }

}
