package be.cylab.mongomail.bizz;

import static be.cylab.mongomail.bizz.Util.*;

/**
 * RCPT Command. See more on SMTP Protocol document.
 *
 * @author Bunyamin Aslan
 */
class CommandRcpt extends Command {

    private String recipient_mail;

    /**
     * Constructor of RcptCommand.
     *
     * @param client
     */
    public CommandRcpt(final IClient client) {
        super(client);
    }

    /**
     * Step 1 : Retrieve the address. Step 2 : Verify if the address is valid.
     * Step 3 : Verify if the address isn't already in the list of recipients,
     * if not add it to the list of recipients Step 4 : Respond "250 OK"
     *
     * @param client_line from client
     */
    @Override
    public void execute(final String client_line) {
        if (!retrieveAddress(client_line)) {
            super.getClient().sendReply(new Reply(
                    ReplyCode.SYNTAX_ERROR,
                    "Syntax: RCPT TO:<recipient@domain.com>"));
            return;
        }
        if (!verifyAddress()) {
            super.getClient().sendReply(new Reply(
                    ReplyCode.INVALID_MAILBOX_SYNTAX,
                    "Invalid email address: " + this.recipient_mail));
            return;
        }
        verifyAddressPresenceAndAdd();
        respond();
    }

    /**
     * Retrieve the address.
     *
     * @param client_line from the client
     * @return true if address retrieved, false otherwise
     */
    private boolean retrieveAddress(final String client_line) {
        String[] args = client_line.split(SPACE_REGEX);
        if (args.length < 2) {
            return false;
        }
        String mail_part = args[1].replace('<', ' ').replace('>', ' ');
        args = mail_part.split(SPACE_REGEX);
        if (args.length < 2) {
            return false;
        }
        this.recipient_mail = args[1];
        return true;
    }

    /**
     * check mail address.
     *
     * @return true if mail is correct, false otherwise
     */
    private boolean verifyAddress() {
        return checkEmail(this.recipient_mail);
    }

    /**
     * Add the recipient to the list of recipients.
     *
     * @return true if the recipient is added, false if the recipient was
     * already in the list
     */
    private boolean verifyAddressPresenceAndAdd() {
        return super.getClient().getMail().addRecipient(this.recipient_mail);
    }

    /**
     * Respond "250 OK".
     */
    private void respond() {
        super.getClient().sendReply(new Reply(ReplyCode.OK, "OK"));
    }

}
