package be.cylab.mongomail.bizz;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Enumeration;
import javax.mail.BodyPart;
import javax.mail.Header;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import org.bson.Document;

/**
 * Get a mail content in string format, parse it to Mime, then parse it to
 * Document bson.
 *
 * @author Bunyamin Aslan
 */
class MimeParser {

    /**
     * "BODY" string.
     */
    public static final String BODY = "BODY";
    /**
     * "FILENAME" string.
     */
    public static final String FILENAME = "FILENAME";
    /**
     * "TO" string.
     */
    private static final String TO = "TO";
    /**
     * "CC" string.
     */
    private static final String CC = "CC";
    /**
     * "BCC" string.
     */
    private static final String BCC = "BCC";
    /**
     * "CONTENT" string.
     */
    private static final String CONTENT = "CONTENT";
    /**
     * "REPLY-TO" string.
     */
    private static final String REPLY_TO = "REPLY-TO";
    /**
     * "RESENT-TO" string.
     */
    private static final String RESENT_TO = "RESENT-TO";
    /**
     * "RESENT-CC" string.
     */
    private static final String RESENT_CC = "RESENT-CC";
    /**
     * "RESENT-BCC" string.
     */
    private static final String RESENT_BCC = "RESENT-BCC";

    private MimeMessage mime;
    private Document mime_json;
    private IMailDto mail;

    /**
     * Constructor of MimeParser. Convert the content of the mail into a MIME.
     *
     * @param mail
     */
    public MimeParser(final IMailDto mail) {
        this.mime_json = new Document();
        this.mail = mail;
        try {
            mime = new MimeMessage(
                    Session.getInstance(System.getProperties(), null),
                    new ByteArrayInputStream(mail.getContent().getBytes()));
        } catch (MessagingException exc) {
            throw new MailParsingException("Mail data cannot be converted "
                    + "into mime. Bad mail content.", exc);
        }
    }

    /**
     * Convert the mime into a document storable in mongoDb. Handle errors from
     * private methods
     *
     */
    public void convertToDocument() {
        try {
            this.parseHeaders(this.mime.getAllHeaders(), this.mime_json);
            this.parseBody();
        } catch (IOException | MessagingException exc) {
            throw new MailParsingException("Mail data cannot be parsed "
                    + "into document", exc);
        }
        this.mail.setDocument(mime_json);
    }

    /**
     * Parses body. Calls handle methods.
     *
     * @throws IOException handled by convertToJson
     * @throws MessagingException handled by convertToJson
     */
    private void parseBody() throws IOException, MessagingException {
        try {
            MimeMultipart multipart = (MimeMultipart) this.mime.getContent();
            handleMultipartMime(multipart);
        } catch (ClassCastException exc) { //If the mime isnt multipart
            handleSimpleMime();
        }
    }

    /**
     * Parses body. Each bodypart has his headers parsed. The attachments are
     * stored in byte[] format. That will be handled by IMailDao because of
     * GridFS.
     *
     * @param multipart
     * @throws MessagingException caught by convertToDocument
     * @throws IOException caught by convertToDocument
     */
    private void handleMultipartMime(final MimeMultipart multipart)
            throws MessagingException, IOException {
        ArrayList<Document> body = new ArrayList<>();
        int multipart_count = multipart.getCount();
        //For each part
        for (int i = 0; i < multipart_count; i++) {
            BodyPart bodypart = multipart.getBodyPart(i);
            Document bodypart_document = new Document();
            this.parseHeaders(bodypart.getAllHeaders(), bodypart_document);
            if (bodypart.isMimeType("text/*")) { //Text
                bodypart_document.put(CONTENT, (String) bodypart.getContent());
            } else { //Attachment (convert inputStream into byte[]
                InputStream inputstream = bodypart.getInputStream();
                byte[] data = new byte[inputstream.available()];
                ByteArrayOutputStream buffer = new ByteArrayOutputStream();
                int bytes_read;
                while ((bytes_read = inputstream.read(data, 0, data.length))
                        != -1) {
                    buffer.write(data, 0, bytes_read);
                }
                this.mail.addAttachment(data);
                bodypart_document.put(
                        FILENAME, parseFilename(bodypart.getFileName()));
            }
            //We add this part to the body
            body.add(bodypart_document);
        }
        //We add body on the mime document
        this.mime_json.put(BODY, body);
    }

    /**
     * Handling simple mail with one part. We have the same structure of
     * handleMultipartMime to keep the same document structure
     *
     * @throws MessagingException
     * @throws IOException
     */
    private void handleSimpleMime() throws MessagingException, IOException {
        ArrayList<Document> body = new ArrayList<>();
        Document bodypart_document = new Document();
        bodypart_document.put(CONTENT, (String) this.mime.getContent());
        body.add(bodypart_document);
        this.mime_json.put(BODY, body);
    }

    /**
     * Parses headers of the mime Default in switch are the non composed
     * headers. Composed headers need to be specified by just adding "case
     * COMPOSED_HEADER:" (1 code line)
     *
     * @param header_enum the headers to parse and convertToJson
     * @param document is the document where we put this headers once parsed
     * @throws MessagingException to be handled by convertToJson
     */
    private void parseHeaders(
            final Enumeration<Header> header_enum, final Document document)
            throws MessagingException {
        while (header_enum.hasMoreElements()) {
            Header h = header_enum.nextElement();
            //cases are when we must split the value into separate tags
            switch (h.getName().toUpperCase()) {
                case REPLY_TO:
                case RESENT_CC:
                case RESENT_TO:
                case RESENT_BCC:
                case CC:
                case BCC:
                case TO:
                    parseComposedHeader(
                            h.getName().toUpperCase(), h.getValue(), document);
                    break;
                default: //The header name in uppercase is the tag
                    document.put(h.getName().toUpperCase(), h.getValue());
                    break;
            }
        }
    }

    /**
     * Parses a composed header like TO or CC. Separate the header between each
     * mail adresses and classify them by counting (TO1, TO2, ...)
     *
     * @param header that we are parsing
     * @param value of the header
     * @param document is the document where we put this headers once parsed
     */
    private void parseComposedHeader(
            final String header, final String value, final Document document) {
        ArrayList<String> composed_header = new ArrayList<>();
        String[] values = value.split(", ");
        for (int i = 0; i < values.length; i++) {
            composed_header.add(values[i]);
        }
        document.put(header, composed_header);
    }

    /**
     * filename is the filename with route (directories). Parse the filename to
     * get just the filename.
     *
     * @param filename with repertories
     * @return the filename without repertories
     */
    private String parseFilename(final String filename) {
        String[] files = filename.split("/");
        return files[files.length - 1];
    }

}
