package be.cylab.mongomail.bizz;

import be.cylab.mongomail.db.IMailDao;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.net.Socket;
import org.apache.logging.log4j.LogManager;

/**
 * Connection is the orchester master. It uses all application services (bizz
 * package) to accomplish the objective of a connection.
 *
 * @author Bunyamin Aslan
 */
public class Connection implements Runnable {

    private final Socket client_socket;
    private final IMailDao mail_dao;
    private final Client client;
    private final SmtpProtocol protocol;

    /**
     * Constructor of SmtpConnection.
     *
     * @param socket client_socket
     * @param mail_dao data access object
     */
    public Connection(
            final Socket socket,
            final IMailDao mail_dao) {
        this.client_socket = socket;
        this.mail_dao = mail_dao;
        this.client = new Client();
        this.client.setMail(new Mail());
        this.protocol = new SmtpProtocol(this.client);
    }

    /**
     * Let the SmtpProtocol handle all comunication with the client. Close the
     * socket and register the mail if it's valid.
     *
     */
    @Override
    public void run() {
        try (
                PrintWriter out = new PrintWriter(
                        client_socket.getOutputStream());
                BufferedReader in = new BufferedReader(
                        new InputStreamReader(
                                client_socket.getInputStream()))) {
            out.flush(); //Dont know if its necessary
            this.client.setIn(in);
            this.client.setOut(out);
            this.protocol.handleProtocol();
            LogManager.getLogger(Connection.class).info(
                    "client " + client_socket.getInetAddress().getHostAddress()
                    + " - Mail correcty received by the protocol");
        } catch (IOException exc) {
            LogManager.getLogger(Connection.class).warn(
                    "client " + client_socket.getInetAddress().getHostAddress()
                    + " - IOException with socket.");
        } catch (ProtocolException exc) {
            LogManager.getLogger(Connection.class).warn(
                    "client " + client_socket.getInetAddress().getHostAddress()
                    + " - " + exc.getMessage());
        } finally {
            closeSocket();
        }
        if (client.getMail().validateMailAfterProtocol()) {
            parseAndRegisterMail();
        }
    }

    /**
     * Let the IMimeParser parse the mail. Then let IMailDao store the mail in
     * the database
     */
    private void parseAndRegisterMail() {
        try {
            MimeParser mime_parser = new MimeParser((this.client.getMail()));
            mime_parser.convertToDocument();
            this.mail_dao.insertMime(this.client.getMail());
            LogManager.getLogger(Connection.class).info(
                    "client " + client_socket.getInetAddress().getHostAddress()
                    + " - Mail correctly parsed and stored.");
        } catch (MailParsingException exc) {
            LogManager.getLogger(Connection.class).warn(
                    "client " + client_socket.getInetAddress().getHostAddress()
                    + " - " + exc.getMessage());
        }
    }

    /**
     * Close the client socket.
     */
    private void closeSocket() {
        try {
            this.client_socket.close();
        } catch (IOException exc) {
            LogManager.getLogger(Connection.class).warn(
                    "client " + client_socket.getInetAddress().getHostAddress()
                    + " - Error while closing socket.");
        }
    }

}
