package be.cylab.mongomail.server;

import io.github.cdimascio.dotenv.Dotenv;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import org.apache.logging.log4j.LogManager;

/**
 * Configuration is working with .env file. It also provide a dependecies
 * injection.
 *
 * @author Bunyamin Aslan
 */
class Configuration implements IConfiguration {

    private final Map<Class, Object> cache_singletons;
    private final Properties prop;

    private final Dotenv dotenv;

    /**
     * Constructor of Configuration.
     *
     * @param dotenv_path
     */
    public Configuration(final String dotenv_path) {
        cache_singletons = new HashMap<>();
        cache_singletons.put(IConfiguration.class, this);
        this.dotenv = Dotenv.configure().directory(dotenv_path).load();
        prop = new Properties();
        InputStream in;
        try {
            in = new FileInputStream("injection.properties");
            prop.load(in);
        } catch (FileNotFoundException exc) {
            LogManager.getLogger(Server.class).fatal(
                    "Cannot load properties file ! "
                    + "Verify if injection.properties "
                    + "is correctly on the project");
        } catch (IOException exc) {
            LogManager.getLogger(Server.class).fatal(
                    "Cannot load properties file ! "
                    + "Verify if injection.properties "
                    + "is correctly on the project");
        }
    }

    @Override
    public String getString(final String key) {
        return dotenv.get(key);
    }

    @Override
    public int getInt(final String key) {
        if (dotenv.get(key) == null) {
            return -1;
        }
        return Integer.parseInt(dotenv.get(key));
    }

    /**
     * Returns an instance of the class implementing the interface given as
     * parameter.
     *
     *
     * @param interface_of_the_instance an interface whose simple name is a key
     * in the .env file
     * @return an instance of the class implementing the interfaceOfTheInstance
     * interface
     */
    public Object getImplementingInstance(
            final Class interface_of_the_instance) {
        if (cache_singletons.containsKey(interface_of_the_instance)) {
            return cache_singletons.get(interface_of_the_instance);
        }
        String impl_name = prop.getProperty(
                interface_of_the_instance.getSimpleName());

        try {
            Constructor<?> constructor = Class.forName(impl_name).
                    getDeclaredConstructors()[0];
            constructor.setAccessible(true);
            Class<?>[] parameter_list = constructor.getParameterTypes();
            List<Object> parameter_instanciated = new ArrayList<>();

            for (Class<?> current_class : parameter_list) {
                parameter_instanciated.add(getImplementingInstance(
                        current_class));
            }

            Object instance = constructor.newInstance(
                    parameter_instanciated.toArray());
            cache_singletons.put(interface_of_the_instance, instance);
            return instance;

        } catch (ClassNotFoundException | IllegalAccessException
                | IllegalArgumentException | InstantiationException
                | SecurityException | InvocationTargetException exc) {
            LogManager.getLogger(Server.class).fatal(
                    "Want to instanciate inexisting command. "
                    + "Normally never happen !");
        }
        return null;
    }

}
