/*
 * Copyright 2014-2018 Rudy De Busscher (https://www.atbash.be)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package be.atbash.util;

/**
 * Utilities around proxied classes. Geared towards to Proxies generated by (CDI) libraries.
 */
public final class ProxyUtils {

    private ProxyUtils() {
    }

    /**
     * Returns the 'real' Class of the proxy by returning the superclass of the proxy. When the class in the parameter
     * isn't a proxy, the class itself is returned.
     * @param currentClass Class to get 'real' class from.
     * @return 'real' Class of the proxy
     */
    public static Class<?> getUnproxiedClass(Class<?> currentClass) {
        if (isProxiedClass(currentClass)) {
            return currentClass.getSuperclass();
        }
        return currentClass;
    }

    /**
     * Returns the 'real' class name of the proxy by returning the super class name of the proxy. When the class in the parameter
     * isn't a proxy, the name of the class itself is returned.
     * @param currentClass Class to get 'real' class name from.
     * @return 'real' Class name of the proxy
     */
    public static String getClassName(Class<?> currentClass) {
        return getUnproxiedClass(currentClass).getName();
    }

    /**
     * Test whether the class is a proxy class by inspecting the name.
     * @param currentClass Class to test
     * @return true if it is a proxy class.
     */
    public static boolean isProxiedClass(Class<?> currentClass) {
        return currentClass.getName().contains("$$EnhancerByCGLIB$$") ||
                currentClass.getName().contains("$$FastClassByCGLIB$$") ||
                currentClass.getName().contains("_$$_javassist") ||
                currentClass.getName().contains("$Proxy$_$$_Weld")||
                currentClass.getName().contains("$$Owb");
    }

}
