package axeldjaha.sms;

import android.Manifest;
import android.app.Activity;
import android.app.ProgressDialog;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.support.v7.app.AppCompatActivity;
import android.telephony.SubscriptionInfo;
import android.telephony.SubscriptionManager;
import android.text.TextUtils;
import android.widget.Toast;

import java.util.ArrayList;
import java.util.List;

import axeldjaha.permission.ADPermission;
import axeldjaha.permission.Interface.OnADPermissionListener;
import axeldjaha.sms.Interface.ADSMSListener;
import axeldjaha.sms.Interface.OnSimCardListener;
import axeldjaha.sms.exception.MessageException;
import axeldjaha.sms.exception.PhoneNumberException;
import axeldjaha.sms.fragment.SimDialogFragment;
import axeldjaha.sms.model.SIM;
import axeldjaha.sms.model.SMS;
import axeldjaha.sms.service.SendSmsService;

/**
 * Created by AXEL DJAHA on 07/05/2017.
 */

public final class ADSMS
{
    private static String TAG = "SMSManager";

    private final int PERMISSION_REQUEST = 402;

    //activité parente
    private AppCompatActivity activity;

    //SMS à envoyer
    private SMS sms;
    private String message;
    private String phoneNumber;

    //callback
    private ADSMSListener listener;

    //pemettra d'adresser les messages de broadcast uniquement à activity
    private String packageName;

    //sim sélectionnée
    private SIM selectedSIM;

    //permissions pour les Android version >= 6
    private  String[] permissions;

    //progressdialog
    private  ProgressDialog progressDialog;

    //Informe le listener du résultat de l'envoi du sms
    private BroadcastReceiver smsSentReceiver = new BroadcastReceiver()
    {
        @Override
        public void onReceive(Context context, Intent intent)
        {
            progressDialog.dismiss();

            switch (getResultCode())
            {
                case Activity.RESULT_OK:
                {
                    listener.onADSmsSucces();
                    break;
                }
                default:
                {
                    listener.onADSmsFailed();
                    break;
                }
            }
        }
    };

    //constructeur privé
    private ADSMS(AppCompatActivity activity)
    {
        this.activity = activity;
        this.packageName = activity.getPackageName();
        this.registerReceiver();

        progressDialog = new ProgressDialog(activity);
        progressDialog.setMessage("Envoi en cours...");
        progressDialog.setCancelable(false);

        permissions = new String[]{
                Manifest.permission.SEND_SMS,
                Manifest.permission.READ_PHONE_STATE
        };
    }

    /**
     * Créer l'instance de ADSMS qui permettra d'envoyer un SMS.
     * @param activity instance de AppCompatActivity
     * @return l'instance de ADSMS créée
     */
    public static ADSMS newInstance(AppCompatActivity activity)
    {
        return new ADSMS(activity);
    }

    /**
     * Définir le message à envoyer
     * @param message message à envoyer
     * @return l'instance de ADSMS créée
     */
    public ADSMS bindMessage(String message)
    {
        this.message = message;
        return this;
    }

    /**
     * Définir le numéro destinataire du SMS à envoyer
     * @param phoneNumber numéro destinataire
     * @return l'instance de ADSMS créée
     */
    public ADSMS bindPhoneNumber(String phoneNumber)
    {
        this.phoneNumber = phoneNumber;
        return this;
    }

    /**
     * Définir un écouteur d'évènement pour l'envoi du SMS
     * @param listener écouteur d'évènement (callback), permettant de savoir si l'envoi du SMS est réussi ou pas
     * @return l'instance de ADSMS créée
     */
    public ADSMS bindListener(ADSMSListener listener)
    {
        this.listener = listener;
        return this;
    }

    /**
     * Méthode d'envoi du SMS
     */
    public void send()
    {
        if(activity == null)
        {
            throw new NullPointerException("ADSMS n'est liée à aucune instance d'une activité");
        }
        else if(TextUtils.isEmpty(message))
        {
            throw new MessageException("Aucun message défini pour l'envoi du SMS\n" +
                    "Appelez bindMessage() sur l'instance de ADSMS");
        }
        else if(TextUtils.isEmpty(phoneNumber))
        {
            throw new PhoneNumberException("Numéro destinataire non défini\n" +
                    "Appelez bindPhoneNumber() sur l'instance de ADSMS");
        }
        else if(phoneNumber.length() != 8)
        {
            throw new PhoneNumberException("Numéro destinataire incorrect\n" +
                    "Le numéro doit être composé de 8 chiffres");
        }
        else if(listener == null)
        {
            throw new NullPointerException("ADSMS n'est liée à aucun écouteur d'évènement\n" +
                    "Appelez bindListener() sur l'instance de ADSMS");
        }

        sms = new SMS();
        sms.setMessage(message);
        sms.setPhoneNumber(phoneNumber);

        ADPermission.check(activity, permissions, new OnADPermissionListener() {
            @Override
            public void onPermissionGranted() {
                openSimCardDialog();
            }

            @Override
            public void onPermissionDenied() {
                Toast.makeText(activity, "Acceptez les autorisations pour continuer", Toast.LENGTH_SHORT).show();
            }
        });
    }

    /**
     * Identique à Activity.registerReceiver
     */
    public void registerReceiver()
    {
        IntentFilter intentFilter = new IntentFilter(packageName);
        activity.registerReceiver(smsSentReceiver, intentFilter);
    }

    /**
     * Choisir la SIM d'envoi du SMS
     */
    private void openSimCardDialog()
    {
        /**
         * Récupération des cartes SIM
         */
        ArrayList<SIM> SIMCardArrayList = new ArrayList<>();
        SubscriptionManager subscriptionManager;
        subscriptionManager = SubscriptionManager.from(activity);
        final List<SubscriptionInfo> subscriptionInfoList = subscriptionManager
                .getActiveSubscriptionInfoList();
        for (SubscriptionInfo subscriptionInfo : subscriptionInfoList)
        {
            SIM sim = new SIM();
            sim.setId(subscriptionInfo.getSubscriptionId());
            sim.setName(subscriptionInfo.getDisplayName().toString());
            SIMCardArrayList.add(sim);
        }

        SimDialogFragment.newIntance(SIMCardArrayList, new OnSimCardListener()
        {
            @Override
            public void onSimCardSelected(SIM SIM)
            {
                selectedSIM = SIM;
                startService();
            }
        }).show(activity.getSupportFragmentManager(), "SimDialogFragment");
    }

    /**
     *
     * Démarrage du service d'envoi du SMS
     */
    private void startService()
    {
        progressDialog.show();

        Intent intent = new Intent(activity, SendSmsService.class);
        intent.putExtra("sms", sms);
        intent.putExtra("sim", selectedSIM);
        intent.putExtra("packageName", packageName);

        //utilisation de service pour l'envoi du SMS
        activity.startService(intent);
    }
}
