package axeldjaha.dialog;

import android.content.Context;
import android.graphics.drawable.Drawable;
import android.support.v4.app.FragmentManager;
import android.text.TextUtils;

import axeldjaha.dialog.Interface.ADDialogListener;
import axeldjaha.dialog.exception.ADDialogException;
import axeldjaha.dialog.fragment.ADDialogFragment;

/**
 * Created by AXEL DJAHA on 30/11/2017
 */

public class ADDialog
{
    private static final String TAG = "ADDialog";
    //contexte
    private Context context;

    //boîte de dialogue
    private ADDialogFragment dialogFragment;

    //titre
    private String headerText;

    //bodyText
    private String bodyText;

    //texte bouton positif
    private String positiveButtonText;

    //texte bouton négatif
    private String negativeButtonText;

    //texte du bouton neutre
    private String neutralButtonText;

    //auto dismiss
    private boolean dialogCancelable;

    //écouteur d'évènement
    private ADDialogListener listener;

    //action à exécuter
    private String action;

    private FragmentManager fragmentManager;

    /**
     * Définir l'icon de l'entête
     * @param drawable
     * @return l'instance de ADDialog créée
     */
    public ADDialog setHeaderIcon(Drawable drawable) {
        dialogFragment.setHeaderIcon(drawable);
        dialogFragment.setHasHeaderIcon(true);
        return this;
    }

    /**
     * Définir la couleur de texte de l'entête
     * @param colorId color resource id
     * @return l'instance de ADDialog créée
     */
    public ADDialog setHeaderTextColor(int colorId) {
        dialogFragment.setHeaderTextColor(colorId);
        dialogFragment.setHasHeaderTextColor(true);
        return this;
    }

    /**
     * Définir la couleur de fond de l'entête
     * @param colorId color resource id
     * @return l'instance de ADDialog créée
     */
    public ADDialog setHeaderBackgroundColor(int colorId) {
        dialogFragment.setHeaderBackgroundColor(colorId);
        dialogFragment.setHasHeaderBackgroundColor(true);
        return this;
    }

    /**
     * Définir la couleur de texte du corps de la boîte de dialogue
     * @param colorId color resource id
     * @return l'instance de ADDialog créée
     */
    public ADDialog setBodyTextColor(int colorId) {
        dialogFragment.setBodyTextColor(colorId);
        dialogFragment.setHasBodyTextColor(true);
        return this;
    }

    /**
     * Définir la couleur de fond du corps
     * @param colorId color resource id
     * @return l'instance de ADDialog créée
     */
    public ADDialog setBodyBackgroundColor(int colorId) {
        dialogFragment.setBodyBackgroundColor(colorId);
        dialogFragment.setHasBodyBackgroundColor(true);
        return this;
    }

    /**
     * Définir la couleur de texte du pied de la boîte de dialogue
     * @param colorId color resource id
     * @return l'instance de ADDialog créée
     */
    public ADDialog setFooterTextColor(int colorId) {
        dialogFragment.setFooterTextColor(colorId);
        dialogFragment.setHasFooterTextColor(true);
        return this;
    }

    /**
     * Définir la couleur de fond du pied de la boîte de dialogue
     * @param colorId color resource id
     * @return l'instance de ADDialog créée
     */
    public ADDialog setFooterBackgroundColor(int colorId) {
        dialogFragment.setFooterBackgroundColor(colorId);
        dialogFragment.setHasFooterBackgroundColor(true);
        return this;
    }

    /**
     * Définir la couleur de fond de la boîte de dialogue.
     * @param colorId color resource id
     * @return l'instance de ADDialog créée
     */
    public ADDialog setDialogBackgroundColor(int colorId) {
        dialogFragment.setDialogBackgroundColor(colorId);
        dialogFragment.setHasDialogBackgroundColor(true);
        return this;
    }

    private ADDialog(FragmentManager fragmentManager)
    {
        this.fragmentManager = fragmentManager;
        dialogCancelable = true;
    }

    /**
     * Créer une instance de la boîte de dialogue
     * @param fragmentManager android.support.v4.app.FragmentManager;
     * @return l'instance de ADDialog créée
     */
    public static ADDialog newInstance(FragmentManager fragmentManager)
    {
        ADDialog dialog = new ADDialog(fragmentManager);
        dialog.dialogFragment = new ADDialogFragment();
        dialog.context = dialog.dialogFragment.getContext();
        return dialog;
    }

    /**
     * Définir le texte de l'entête
     * @param headerText
     * @return l'instance de ADDialog créée
     */
    public ADDialog setHeaderText(String headerText) {
        this.headerText = headerText;
        return this;
    }

    /**
     * Définir le texte à afficher dans le corps de la boîte de dialogue
     * @param bodyText
     * @return l'instance de ADDialog créée
     */
    public ADDialog setBodyText(String bodyText) {
        this.bodyText = bodyText;
        return this;
    }

    /**
     * Définir le texte du bouton positif de la bôite de dialogue
     * @param positiveButtonText
     * @return l'instance de ADDialog créée
     */
    public ADDialog setPositiveButtonText(String positiveButtonText) {
        this.positiveButtonText = positiveButtonText;
        return this;
    }

    /**
     * Définir le texte du bouton négatif de la bôite de dialogue
     * @param negativeButtonText
     * @return l'instance de ADDialog créée
     */
    public ADDialog setNegativeButtonText(String negativeButtonText) {
        this.negativeButtonText = negativeButtonText;
        return this;
    }

    /**
     * Définir le texte du bouton neutre de la bôite de dialogue
     * @param neutralButtonText
     * @return l'instance de ADDialog créée
     */
    public ADDialog setNeutralButtonText(String neutralButtonText) {
        this.neutralButtonText = neutralButtonText;
        return this;
    }

    /**
     * Définir le comportement de la boîte de dialogue
     * @param cancelable
     * @return l'instance de ADDialog créée
     */
    public ADDialog setDialogCancelable(boolean cancelable) {
        dialogCancelable = cancelable;
        return this;
    }

    /**
     * Définir l'écouteur d'évènement pour les boutons du pied de la boîte de dialogue
     * @param listener
     * @return l'instance de ADDialog créée
     */
    public ADDialog setListener(ADDialogListener listener) {
        this.listener = listener;
        return this;
    }

    /**
     * Définir l'action à exécuter
     * @param action correspond à un texte permettant de savoir les instructions à exécuter.
     *               Dans le cas où différents éléments peuvent ouvrir la boîte de dialogue,
     *               on pourra savoir les instructions à exécuter en examinant la valeur du
     *               paramètre 'action' des méthodes de l'interface ADDialogListener.
     *               Un élément peut être un Button, un TextView, un EditText ou tout autre widget.
     * @return l'instance de ADDialog créée
     */
    public ADDialog setAction(String action) {
        this.action = action;
        return this;
    }

    /**
     * Afficher la boîte de dialogue
     */
    public void show()
    {
        if(TextUtils.isEmpty(bodyText) && TextUtils.isEmpty(headerText))
        {
            throw new ADDialogException("Titre et bodyText de la boîte de dialogue non définis.\n" +
                    "Définir le titre ou le bodyText de la boîte de dialogue");
        }

        //texte des éléments
        dialogFragment.setHeaderText(headerText);
        dialogFragment.setBodyText(bodyText);
        dialogFragment.setPositiveButtonText(positiveButtonText);
        dialogFragment.setNegativeButtonText(negativeButtonText);
        dialogFragment.setNeutralButtonText(neutralButtonText);

        dialogFragment.setDialogCancelable(dialogCancelable);
        dialogFragment.setAction(TextUtils.isEmpty(action) ? "" : action);
        dialogFragment.setListener(listener);

        dialogFragment.show(fragmentManager, "ADDialogFragment");
    }


    /**
     * Définir le padding de l'entête de la bôite de dialogue
     * @param left Aucun effet si valeur inférieure ou égale à la valeur par defaut (20dp)
     * @param top Aucun effet si valeur inférieure ou égale à la valeur par defaut (10dp)
     * @param right Aucun effet si valeur inférieure ou égale à la valeur par defaut (20dp)
     * @param bottom Aucun effet si valeur inférieure ou égale à la valeur par defaut (10dp)
     * @return l'instance de ADDialog créée
     */
    public ADDialog setHeaderPadding(int left, int top, int right, int bottom)
    {
        dialogFragment.setHeaderPadding(left, top, right, bottom);
        return this;
    }

    /**
     * Définir le padding du corps de la bôite de dialogue
     * @param left Aucun effet si valeur inférieure ou égale à la valeur par defaut (20dp)
     * @param top Aucun effet si valeur inférieure ou égale à la valeur par defaut (0dp)
     * @param right Aucun effet si valeur inférieure ou égale à la valeur par defaut (20dp)
     * @param bottom Aucun effet si valeur inférieure ou égale à la valeur par defaut (10dp)
     * @return l'instance de ADDialog créée
     */
    public ADDialog setBodyPadding(int left, int top, int right, int bottom)
    {
        dialogFragment.setBodyPadding(left, top, right, bottom);
        return this;
    }

    /**
     * Définir le padding du pied de la bôite de dialogue
     * @param left Aucun effet si valeur inférieure ou égale à la valeur par defaut (0dp)
     * @param top Aucun effet si valeur inférieure ou égale à la valeur par defaut (0dp)
     * @param right Aucun effet si valeur inférieure ou égale à la valeur par defaut (0dp)
     * @param bottom Aucun effet si valeur inférieure ou égale à la valeur par defaut (0dp)
     * @return l'instance de ADDialog créée
     */
    public ADDialog setFooterPadding(int left, int top, int right, int bottom)
    {
        dialogFragment.setFooterPadding(left, top, right, bottom);
        return this;
    }
}
