package com.auroral.widget.view;

import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.RectF;
import android.os.Build;
import android.support.annotation.Nullable;
import android.support.annotation.RequiresApi;
import android.util.AttributeSet;
import android.view.View;

/**
 * Created by Auroral 2018/12/8 11:25
 */
public class SquareView extends View {

    private String mText;
    private String mSign;
    private int mBaseWidth;
    private int mRoundAngle;
    private boolean mIsShowSign;
    private RectF mBackgroundRect;
    private Paint mBackgroundPaint, mTextPaint, mSignBackgroundPaint, mSignTextPaint;

    public SquareView(Context context) {
        super(context);
        init(context);
    }

    public SquareView(Context context, @Nullable AttributeSet attrs) {
        super(context, attrs);
        init(context);
    }

    public SquareView(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init(context);
    }

    @RequiresApi(Build.VERSION_CODES.LOLLIPOP)
    public SquareView(Context context, @Nullable AttributeSet attrs, int defStyleAttr, int defStyleRes) {
        super(context, attrs, defStyleAttr, defStyleRes);
        init(context);
    }

    private void init(Context context) {
        mText = "";
        mSign = "";
        // round square background.
        mBackgroundRect = new RectF();
        mBackgroundPaint = new Paint();
        // round square text.
        mTextPaint = new Paint();
        mTextPaint.setTextSize(sp2px(context, 16));
        mTextPaint.setTextAlign(Paint.Align.CENTER);
        // sign background.
        mSignBackgroundPaint = new Paint();
        // sign text.
        mSignTextPaint = new Paint();
        mSignTextPaint.setTextSize(sp2px(context, 10));
        mSignTextPaint.setTextAlign(Paint.Align.CENTER);
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        // set square width equal height.
        super.onMeasure(widthMeasureSpec, widthMeasureSpec);
    }

    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh) {
        super.onSizeChanged(w, h, oldw, oldh);
        // square width.
        mBaseWidth = w;
        // square background width.
        int roomWidth = (int) (mBaseWidth * 0.7);
        int left = mBaseWidth / 2 - roomWidth / 2;
        int right = left + roomWidth;
        // set square position.
        mBackgroundRect.left = left;
        mBackgroundRect.top = left;
        mBackgroundRect.right = right;
        mBackgroundRect.bottom = right;
        // set square round angle.
        mRoundAngle = roomWidth / 7;
    }

    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);
        // round square background.
        canvas.drawRoundRect(mBackgroundRect, mRoundAngle, mRoundAngle, mBackgroundPaint);
        // round square text.
        canvas.drawText(mText,
                mBackgroundRect.centerX(),
                (mBackgroundRect.bottom + mBackgroundRect.top - mTextPaint.getFontMetrics().bottom - mTextPaint.getFontMetrics().top) / 2,
                mTextPaint);

        if (mIsShowSign) {
            // sign background.
            canvas.drawCircle(mBackgroundRect.right, mBackgroundRect.top, (int) (mBaseWidth * 0.6 / 5), mSignBackgroundPaint);
            // sign text.
            canvas.drawText(mSign,
                    mBackgroundRect.right,
                    mBackgroundRect.top + (mSignTextPaint.getFontMetrics().bottom - mSignTextPaint.getFontMetrics().top) / 4,
                    mSignTextPaint);
        }
    }

    /**
     * set square text.
     *
     * @param text            text.
     * @param backgroundColor background color.
     * @param textColor       text color.
     */
    public void setText(String text, int backgroundColor, int textColor) {
        mText = text;
        mBackgroundPaint.setColor(backgroundColor);
        mTextPaint.setColor(textColor);
    }

    /**
     * set square sign.
     *
     * @param sign            sign
     * @param backgroundColor background color.
     * @param textColor       text color.
     */
    public void setSign(String sign, int backgroundColor, int textColor) {
        mIsShowSign = true;
        mSign = sign;
        mSignBackgroundPaint.setColor(backgroundColor);
        mSignTextPaint.setColor(textColor);
    }

    /**
     * sp to px.
     *
     * @param context Context
     * @param sp      sp
     * @return px
     */
    private int sp2px(Context context, int sp) {
        float fontScale = context.getResources().getDisplayMetrics().scaledDensity;
        return (int) (sp * fontScale + 0.5f);
    }
}
