/*
 * Copyright 2013 Coherent Software Australia Pty Ltd (http://www.coherentsoftware.com.au)
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */



package au.com.coherentsoftware.gradle.database.task

import groovy.sql.Sql
import org.gradle.api.tasks.TaskAction

/**
 * Gradle Task to reset a database by dropping it then re-creating it.
 *
 * Database user must have appropriate permissions.
 *
 * See the DatabasePlugin documentation for sample use.
 *
 * @see au.com.coherentsoftware.gradle.DatabasePlugin
 */
class DbReset extends DbBaseTask {
    List<String> databaseList = []
    String dbType = 'mysql'

    /**
     * Add a database name to the list of databases to be reset.
     * @param dbname The name of the database to add.
     */
    def setDatabase(String dbname) { databaseList << dbname }

    /**
     * Add a list of database names to those to be reset.
     * @param dbnamelist A list of database names
     */
    def setDatabases(List<String> dbnamelist) { databaseList += dbnamelist }

    /**
     * A dynamically generated description of this task, which includes the names of the databases it will reset.
     * @return A description.
     */
    String getDescription() {
        "Drops and re-creates databases ${databaseList}".toString()
    }

    /**
     * Drops and re-creates each database.
     * Logs at the lifecycle level the name of each database it resets.
     */
    @TaskAction
    def dbReset() {
        def url = "jdbc:${dbType}://${host}:${port}/"
        Properties props = [user: username, password: password] as Properties

        Sql sql = Sql.newInstance(url, props, driver)
        databaseList.each { dbName ->
            logger.info("Db reset: ${dbName}")
            sql.execute("drop database if exists ${dbName}".toString())
            sql.execute("create database ${dbName}".toString())
        }
        sql.close()
    }

}