/*
 * Copyright 2013 Coherent Software Australia Pty Ltd (http://www.coherentsoftware.com.au)
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package au.com.coherentsoftware.gradle.database.task

import org.gradle.api.tasks.StopExecutionException
import org.gradle.api.tasks.TaskAction

/**
 * Gradle Task to initialise a database from a file containing SQL statements.
 *
 * Database user must have appropriate permissions.
 *
 * See the DatabasePlugin documentation for sample use.
 *
 * @see au.com.coherentsoftware.gradle.DatabasePlugin
 */
class DbInit extends DbBaseTask {
    def dataFile
    String database

    /**
     * Convert datafile to a File object, using Project's file().
     * Ensures interpretation of datafile location is delayed until needed.
     * @return a File object representing the datafile to be loaded.
     */
    File getDataFile() {
        project.file(dataFile)
    }

    /**
     * Customises the description of this task according to the datafile it is to load, and the
     * database it is to update. Makes it easy to work out which task you want to call
     * when there are several different DbInit type tasks.
     * @return A String with the relevant description.
     */
    String getDescription() {
        def relative = getDataFile().absolutePath - project.rootDir.absolutePath - '/'
        "Load ${relative} into database ${database}".toString()
    }

    /**
     * Use the mysql client (which must be on the path) to invoke the SQL statements
     * in the specified datafile against the specified database.
     * The specified username and password must belong to a user who already has permission
     * to take the specified actions on the specified database.
     * The specified database must already exist.
     */
    @TaskAction
    def importSql() {
        File file = getDataFile()
        // delay resolving file location
        if (!file || !file.canRead()) {
            logger.error('No datafile passed or datafile is not readable')
            throw new StopExecutionException()
        }
        logger.info("Loading data from ${file.name} into database ${database}")
        project.exec {
            standardInput = new FileInputStream(file)
            commandLine 'mysql', "--user=${username}", "--password=${password}",
                    "--host=${host}", "--port=${port}", database
        }
    }
}
